jQuery(document).ready(function ($) {
    // ===========================================
    // 全局變數定義
    // ===========================================
    let isGridMode = false;
    let zIndex = 400;
    let cardPositions = [];
    let dragState = {
        isDragging: false,
        draggedElement: null,
        startX: 0,
        startY: 0,
        offsetX: 0,
        offsetY: 0,
        initialX: 0,
        initialY: 0
    };

    const langTexts = {
        zh: { 
            grid: '<i class="fa fa-th" aria-hidden="true"></i>',
            random: '<i class="fa fa-random" aria-hidden="true"></i>'
        },
        en: { 
            grid: '<i class="fa fa-th" aria-hidden="true"></i>',
            random: '<i class="fa fa-random" aria-hidden="true"></i>'
        }
    };
    const currentLang = smcData.currentLang || 'zh';

    // 關鍵DOM元素
    const cardContainer = $("#smc-card-container");
    const cards = $(".smc-card");
    const layoutToggleBtn = $("#smc-layout-toggle-btn");
    const layoutToggleText = $("#smc-toggle-text");
    const layoutToggleIcon = $("#smc-toggle-icon"); 
    const modal = $("#smc-card-modal");
    const modalBody = $("#smc-modal-body");
    const modalClose = $("#smc-modal-close");
    const projectsTab = $("#projects-tab");
    const projectsPanel = $("#project-nav-panel");
    const projectsClose = $("#project-nav-close");
    const languageTab = $("#language-tab");
    const homeTab = $("#smc-home-tab");
    const homeContent = $("#home-content");
    const exhibitionContent = $("#exhibition-content");

    // 工具函數
    const isDesktop = () => window.innerWidth > 768;
    const isMobile = () => window.innerWidth <= 768;

    // 檢查關鍵元素是否存在
    if (cardContainer.length === 0) {
        console.log('SMC: Card container not found');
        return;
    }

    // ===========================================
    // 首頁/展覽內容切換系統
    // ===========================================
    
    /**
     * 顯示首頁內容，隱藏展覽內容
     */
    const showHomeContent = () => {
        if (homeContent.length && exhibitionContent.length) {
            homeContent.show();
            exhibitionContent.hide().html(''); // 清空展覽內容
            
            // 移除可能存在的子項目網格
            $('#sub-items-grid').remove();
            $('#sub-level-grid').remove();
            
            // 滾動到首頁內容區域
            $('html, body').animate({
                scrollTop: homeContent.offset().top - 100
            }, 500);
            
            console.log('Home content displayed');
        } else {
            console.warn('Home or Exhibition content container not found');
        }
    };

    /**
     * 顯示展覽內容，隱藏首頁內容
     */
    const showExhibitionContent = () => {
        if (homeContent.length && exhibitionContent.length) {
            homeContent.hide();
            exhibitionContent.show();
            
            console.log('Exhibition content displayed');
        }
    };

    // ===========================================
    // 1. 卡片隨機分佈系統
    // ===========================================
    // ===========================================
    // 智能卡片隨機分佈系統 - 根據卡片數量自動調整
    // ===========================================

    const generateRandomPositions = () => {
        if (isMobile()) return;
        
        cardPositions = [];
        const containerRect = cardContainer[0].getBoundingClientRect();
        const containerWidth = containerRect.width;
        const containerHeight = containerRect.height;
        const cardWidth = 280;
        const cardHeight = 320;
        
        const visibleCards = cards.filter(':visible');
        const cardCount = visibleCards.length;
        
        // 根據卡片數量動態調整參數
        let minDistance, marginTop, marginBottom, marginLeft, marginRight;
        
        if (cardCount <= 4) {
            // 少量卡片：更分散
            minDistance = 100;
            marginTop = 100;
            marginBottom = 150;
            marginLeft = 50;
            marginRight = 50;
        } else if (cardCount <= 8) {
            // 中等數量：平衡分佈
            minDistance = 50;
            marginTop = 80;
            marginBottom = 120;
            marginLeft = 30;
            marginRight = 30;
        } else {
            // 大量卡片：緊密排列
            minDistance = 25;
            marginTop = 50;
            marginBottom = 80;
            marginLeft = 20;
            marginRight = 20;
        }
        
        // 計算實際可用區域
        const availableWidth = containerWidth - marginLeft - marginRight - cardWidth;
        const availableHeight = containerHeight - marginTop - marginBottom - cardHeight;
        
        // 檢查可用空間是否足夠
        if (availableWidth < 0 || availableHeight < 0) {
            console.warn('Container too small for cards, using fallback layout');
            // 使用簡單的瀑布流佈局
            visibleCards.each((index, card) => {
                const col = index % 3;
                const row = Math.floor(index / 3);
                const x = col * (cardWidth + 20) + 20;
                const y = row * (cardHeight + 20) + 20;
                cardPositions.push({ x: x, y: y, element: $(card) });
            });
            return;
        }

        // 使用改進的網格輔助分佈
        const aspectRatio = containerWidth / containerHeight;
        const cols = Math.ceil(Math.sqrt(cardCount * aspectRatio));
        const rows = Math.ceil(cardCount / cols);
        const cellWidth = availableWidth / cols;
        const cellHeight = availableHeight / rows;
        
        console.log(`Layout: ${cols} cols × ${rows} rows for ${cardCount} cards`);
        
        visibleCards.each((index, card) => {
            let x, y, attempts = 0;
            let validPosition = false;

            // 計算該卡片的網格位置
            const gridRow = Math.floor(index / cols);
            const gridCol = index % cols;
            
            // 網格中心點
            const gridCenterX = marginLeft + gridCol * cellWidth + cellWidth / 2;
            const gridCenterY = marginTop + gridRow * cellHeight + cellHeight / 2;
            
            // 隨機範圍：根據單元格大小調整
            const randomRangeX = Math.min(cellWidth * 0.35, 80);
            const randomRangeY = Math.min(cellHeight * 0.35, 80);

            while (!validPosition && attempts < 30) {
                // 在網格中心附近隨機偏移
                const offsetX = (Math.random() - 0.5) * randomRangeX;
                const offsetY = (Math.random() - 0.5) * randomRangeY;
                
                x = gridCenterX + offsetX;
                y = gridCenterY + offsetY;
                
                // 確保在邊界內
                x = Math.max(marginLeft, Math.min(x, marginLeft + availableWidth));
                y = Math.max(marginTop, Math.min(y, marginTop + availableHeight));

                validPosition = true;
                
                // 檢查與其他卡片的距離
                for (let i = 0; i < cardPositions.length; i++) {
                    const pos = cardPositions[i];
                    const dx = x - pos.x;
                    const dy = y - pos.y;
                    const distance = Math.sqrt(dx * dx + dy * dy);
                    
                    // 動態最小距離
                    const requiredDistance = minDistance + (cardWidth * 0.2);
                    
                    if (distance < requiredDistance) {
                        validPosition = false;
                        break;
                    }
                }
                attempts++;
            }
            
            // 如果隨機分佈失敗，使用網格位置並加入小偏移
            if (!validPosition) {
                const smallOffset = 10;
                x = marginLeft + gridCol * cellWidth + (Math.random() - 0.5) * smallOffset;
                y = marginTop + gridRow * cellHeight + (Math.random() - 0.5) * smallOffset;
            }
            
            // 最終邊界檢查
            x = Math.max(marginLeft, Math.min(x, containerWidth - cardWidth - marginRight));
            y = Math.max(marginTop, Math.min(y, containerHeight - cardHeight - marginBottom));
            
            cardPositions.push({ x: x, y: y, element: $(card) });
        });
        
        console.log(`Successfully positioned ${cardPositions.length} cards`);
    };

    const applyRandomPositions = () => {
        if (isMobile()) return;
        
        cardPositions.forEach((position, index) => {
            if (position.element && 
                position.element.length > 0 && 
                $.contains(document, position.element[0]) &&
                position.element.is(':visible')) {
                
                // 初始狀態：隱藏並下移
                position.element.css({
                    left: position.x + "px",
                    top: position.y + "px",
                    opacity: 0,
                    visibility: 'visible',
                    transform: 'translateY(30px) scale(0.95)'
                });
                
                // 波浪式出場：每張卡片延遲 80ms
                setTimeout(() => {
                    position.element.addClass('animating');
                    
                    // 動畫完成後清理
                    setTimeout(() => {
                        position.element.removeClass('animating').css({
                            opacity: 1,
                            transform: 'none'
                        });
                    }, 600);
                }, index * 80);
            }
        });
        
        console.log(`Applied positions with silky smooth wave animation`);
    };
    // ===========================================
    // 2. 拖拽系統
    // ===========================================
    const enableCardDragging = () => {
        if (isMobile()) return;
        
        cards.off('.drag');
        $(document).off('.dragGlobal');

        cards.on('mousedown.drag touchstart.drag', function (e) {
            if ($(e.target).hasClass('smc-popup-trigger') || $(e.target).closest('.smc-popup-trigger').length) {
                return;
            }

            e.preventDefault();
            
            const card = $(this);
            const clientX = e.clientX || (e.originalEvent.touches && e.originalEvent.touches[0].clientX);
            const clientY = e.clientY || (e.originalEvent.touches && e.originalEvent.touches[0].clientY);
            
            if (!clientX || !clientY) return;

            dragState.draggedElement = card;
            dragState.initialX = clientX;
            dragState.initialY = clientY;
            
            const rect = card[0].getBoundingClientRect();
            dragState.offsetX = clientX - rect.left;
            dragState.offsetY = clientY - rect.top;
            
            let animationId;
            let lastX = clientX;
            let lastY = clientY;

            const updatePosition = () => {
                if (!dragState.isDragging || !dragState.draggedElement) {
                    return;
                }
                
                const moveX = lastX - dragState.offsetX;
                const moveY = lastY - dragState.offsetY;
                
                dragState.draggedElement.css({
                    'left': moveX + 'px',
                    'top': moveY + 'px'
                });
            };

            const handleMove = (e) => {
                if (!dragState.isDragging) {
                    const currentX = e.clientX || (e.originalEvent.touches && e.originalEvent.touches[0].clientX);
                    const currentY = e.clientY || (e.originalEvent.touches && e.originalEvent.touches[0].clientY);
                    
                    const deltaX = currentX - dragState.initialX;
                    const deltaY = currentY - dragState.initialY;
                    const distance = Math.sqrt(deltaX * deltaX + deltaY * deltaY);
                    
                    if (distance > 5) {
                        dragState.isDragging = true;
                        card.addClass('smc-dragging');
                        card.css({
                            'z-index': ++zIndex,
                            'transition': 'none',
                            'pointer-events': 'none'
                        });
                    }
                }
                
                if (dragState.isDragging) {
                    e.preventDefault();
                    
                    lastX = e.clientX || (e.originalEvent.touches && e.originalEvent.touches[0].clientX);
                    lastY = e.clientY || (e.originalEvent.touches && e.originalEvent.touches[0].clientY);
                    
                    if (animationId) {
                        cancelAnimationFrame(animationId);
                    }
                    
                    animationId = requestAnimationFrame(updatePosition);
                }
            };

            $(document).on('mousemove.dragGlobal touchmove.dragGlobal', handleMove);

            $(document).on('mouseup.dragGlobal touchend.dragGlobal', function (e) {
                if (dragState.isDragging && dragState.draggedElement) {
                    if (animationId) {
                        cancelAnimationFrame(animationId);
                    }
                    
                    dragState.draggedElement.css({
                        'pointer-events': 'auto'
                    });
                    
                    dragState.draggedElement.removeClass('smc-dragging');
                }
                
                dragState.isDragging = false;
                dragState.draggedElement = null;
                $(document).off('.dragGlobal');
            });
        });
        
        cards.css({
            'will-change': 'transform',
            'backface-visibility': 'hidden',
            'transform-style': 'preserve-3d',
            'transition': 'none'
        });
    };

    const disableCardDragging = () => {
        cards.css({
            'cursor': 'default',
            'will-change': 'auto',
            'transform': 'none',
            'transition': ''
        });
        cards.off('.drag');
        $(document).off('.dragGlobal');
        cards.removeClass('smc-dragging');
        dragState.isDragging = false;
        dragState.draggedElement = null;
    };

    // ===========================================
    // 3. 佈局切換系統
    // ===========================================
    const toggleLayout = () => {
        if (isMobile()) return;
        
        const button = layoutToggleBtn;
        const icon = layoutToggleIcon;
        
        // 階段1：按鈕脈衝 + 圖標消失
        button.addClass('switching icon-out');
        
        setTimeout(() => {
            // 切換模式
            isGridMode = !isGridMode;
            
            if (isGridMode) {
                cardContainer.addClass("smc-grid-mode");
                icon.html(langTexts[currentLang].random);
                disableCardDragging();
            } else {
                cardContainer.removeClass("smc-grid-mode");
                icon.html(langTexts[currentLang].grid);
                setTimeout(() => {
                    generateRandomPositions();
                    applyRandomPositions();
                    enableCardDragging();
                }, 100);
            }
            
            // 階段2：新圖標出現
            button.removeClass('icon-out').addClass('icon-in');
            
            // 階段3：清理動畫類
            setTimeout(() => {
                button.removeClass('switching icon-in');
            }, 400);
            
        }, 300);
    };

    // ===========================================
    // 4. 子項目網格顯示系統
    // ===========================================
    const createSubItemsGrid = (subItems, parentTitle, chaptersLabel, parentContainer = exhibitionContent) => {
        console.log('createSubItemsGrid called with:', subItems, parentTitle);
        $('#sub-items-grid').remove();
        
        if (!subItems || subItems.length === 0) {
            console.log('No subItems or empty array');
            return;
        }
        
        const gridContainer = $(`
            <div id="sub-items-grid" style="margin-top: 30px; background: rgba(40,40,40,0.8); border: 1px solid rgba(80,80,80,0.6); border-radius: 12px; padding: 20px; backdrop-filter: blur(10px);">
                <h3 style="color: #e0e0e0; margin-bottom: 20px;">${parentTitle} ${chaptersLabel}</h3>
                <div class="sub-items-container" style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 20px;"></div>
            </div>
        `);
        
        parentContainer.append(gridContainer);
        const container = gridContainer.find('.sub-items-container');

        subItems.forEach(item => {
            const itemCard = $(`
                <div class="sub-item-card" data-item-id="${item.id}" data-item-type="${item.type}" 
                     style="height:fit-content; background: #333; border: 1px solid rgba(255,255,255,0.2); border-radius: 8px; overflow: hidden; transition: all 0.3s ease; position: relative;">
                    ${item.thumbnail ? `
                        <div style="width: 100%;  overflow: hidden; display: flex; align-items: center; justify-content: center; background: #222;">
                            <img src="${item.thumbnail}" alt="${item.title}" style="max-width: 100%;  width: 100%; height: auto; object-fit: contain;">
                        </div>
                    ` : ''}
                    <div style="padding: 15px; padding-bottom: 50px;">
                        <h4 style="color: #e0e0e0; margin: 0 0 10px 0;">${item.title}</h4>
                        ${item.has_children ? `
                            <div style="color: rgba(255,255,255,0.5); font-size: 11px; margin-top: 5px;">${item.children.length} 個子項目</div>
                        ` : ''}
                    </div>
                    <div style="position: absolute; bottom: 10px; right: 10px; display: flex; gap: 6px;">
                        <button class="view-content-btn" style="width: 32px; height: 32px; background: rgba(0, 0, 0, 0.8); border: 1px solid rgba(255,255,255,0.7); color: rgba(255,255,255,1); border-radius: 50%; cursor: pointer; display: flex; align-items: center; justify-content: center; transition: all 0.3s ease;" title="查看內容 / View Content">
                            <i class="fa fa-eye" aria-hidden="true" style="font-size: 12px;"></i>
                        </button>
                        ${item.has_children ? `
                            <button class="expand-children-btn" style="width: 32px; height: 32px; background: rgba(255, 255, 255, 0.8); border: 1px solid rgba(0, 0, 0, 0.2); color: #000; border-radius: 50%; cursor: pointer; display: flex; align-items: center; justify-content: center; transition: all 0.3s ease;" title="展開項目 / Expand Items">
                                <i class="fa fa-chevron-down" aria-hidden="true" style="font-size: 12px;"></i>
                            </button>
                        ` : ''}
                    </div>
                </div>
            `);
            
            itemCard.find('.view-content-btn').on('click', function(e) {
                e.stopPropagation();
                const itemId = itemCard.data('item-id');
                const itemType = itemCard.data('item-type');
                
                if (itemType === 'post') {
                    showModal(itemId, null, true);
                } else {
                    showModal(null, itemId, true);
                }
            });
            
            if (item.has_children && item.children.length > 0) {
                itemCard.find('.expand-children-btn').on('click', function(e) {
                    e.stopPropagation();
                    showSubLevel(item);
                });
                
                itemCard.on('click', function(e) {
                    if (!$(e.target).hasClass('view-content-btn') && 
                        !$(e.target).hasClass('expand-children-btn') && 
                        !$(e.target).closest('.view-content-btn, .expand-children-btn').length) {
                        showSubLevel(item);
                    }
                });
            } else {
                itemCard.on('click', function(e) {
                    if (!$(e.target).hasClass('view-content-btn') && 
                        !$(e.target).closest('.view-content-btn').length) {
                        const itemId = $(this).data('item-id');
                        const itemType = $(this).data('item-type');
                        
                        if (itemType === 'post') {
                            showModal(itemId, null, true);
                        } else {
                            showModal(null, itemId, true);
                        }
                    }
                });
            }
            
            itemCard.on('mouseenter', function() {
                $(this).css({
                    'transform': 'translateY(-3px)',
                    'box-shadow': '0 8px 16px rgba(0,0,0,0.4)',
                    'background': '#555'
                });
            });
            
            itemCard.on('mouseleave', function() {
                $(this).css({
                    'transform': 'translateY(0)',
                    'box-shadow': 'none',
                    'background': '#333'
                });
            });
            
            container.append(itemCard);
        });
        
        const updateGridColumns = () => {
            const containerWidth = container.width();
            let columns = Math.floor(containerWidth / 270);
            columns = Math.max(1, Math.min(columns, 4));
            container.css('grid-template-columns', `repeat(${columns}, 1fr)`);
        };
        
        updateGridColumns();
        $(window).on('resize', updateGridColumns);
    };

    const showSubLevel = (parentItem) => {
        $('#sub-level-grid').remove();
        
        const subLevelContainer = $(`
            <div id="sub-level-grid" style="margin-top: 20px; background: rgba(20,20,20,0.6); border: 1px solid rgba(60,60,60,0.6); border-radius: 8px; padding: 15px;">
                <div style="display: flex; align-items: center; justify-content: flex-end; margin-bottom: 15px;">
                    <button id="close-sub-level" style="background: rgba(255,255,255,0.1); border: 1px solid rgba(255,255,255,0.7); color: #e0e0e0; padding: 5px 10px; border-radius: 4px; cursor: pointer; font-size: 12px;">返回</button>
                </div>
                <div class="sub-level-container" style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px;"></div>
            </div>
        `);
        
        $('#sub-items-grid').after(subLevelContainer);
        const container = subLevelContainer.find('.sub-level-container');
        
        parentItem.children.forEach(child => {
            const childCard = $(`
                <div class="sub-level-card" data-post-id="${child.id}" 
                     style="height:fit-content; background: #2a2a2a; border: 1px solid rgba(255,255,255,0.15); border-radius: 6px; overflow: hidden; cursor: pointer; transition: all 0.3s ease; position: relative;">
                    ${child.thumbnail ? `
                        <div style="width: 100%; overflow: hidden; display: flex; align-items: center; justify-content: center; background: #222;">
                            <img src="${child.thumbnail}" alt="${child.title}" style="max-width: 100%; width: 100%; height: auto; object-fit: contain;">
                        </div>
                    ` : ''}
                    <div style="padding: 12px; padding-bottom: 50px;">
                        <h5 style="color: #e0e0e0;">${child.title}</h5>
                    </div>
                    <div style="position: absolute; bottom: 10px; right: 10px;">
                        <button class="view-post-btn" style="width: 32px; height: 32px; background: rgba(0, 0, 0, 0.8); border: 1px solid rgba(255,255,255,0.7); color: rgba(255,255,255,1); border-radius: 50%; cursor: pointer; display: flex; align-items: center; justify-content: center; transition: all 0.3s ease;" title="查看內容 / View Content">
                            <i class="fa fa-eye" aria-hidden="true" style="font-size: 12px;"></i>
                        </button>
                    </div>
                </div>
            `);

            childCard.find('.view-post-btn').on('click', function(e) {
                e.stopPropagation();
                const postId = childCard.data('post-id');
                showModal(postId, null, true);
            });

            childCard.on('mouseenter', function() {
                $(this).css({
                    'transform': 'translateY(-2px)',
                    'box-shadow': '0 4px 8px rgba(0,0,0,0.3)',
                    'background': '#3a3a3a'
                });
            });

            childCard.on('mouseleave', function() {
                $(this).css({
                    'transform': 'translateY(0)',
                    'box-shadow': 'none',
                    'background': '#2a2a2a'
                });
            });
            
            container.append(childCard);
        });
        
        $('#close-sub-level').on('click', function() {
            $('#sub-level-grid').fadeOut(300, function() {
                $(this).remove();
            });
        });
        
        $('html, body').animate({
            scrollTop: subLevelContainer.offset().top - 20
        }, 300);
    };

    // ===========================================
    // 5. 內容替換與Modal系統
    // ===========================================
    const showModal = (postId, categoryId = null, forceModal = false) => {
        if (!postId && !categoryId) return;

        $(".smc-nav-panel.open").removeClass("open");
        $('body').removeClass('smc-panel-open');

        const ajaxAction = categoryId ? 'get_category_content' : 'get_post_content';
        const ajaxData = {
            action: ajaxAction,
            nonce: smcData.nonce
        };
        
        if (categoryId) {
            ajaxData.category_id = categoryId;
        } else {
            ajaxData.post_id = postId;
        }

        // 判斷是否應該替換內容（使用 exhibitionContent）
        const shouldReplaceContent = !forceModal && categoryId && exhibitionContent.length > 0;
        
        if (shouldReplaceContent) {
            // 切換到展覽內容區域
            showExhibitionContent();
            exhibitionContent.html('<div style="text-align:center; padding:40px; color:#e0e0e0;">載入中...</div>');
        } else {
            modalBody.html('<div style="text-align:center; padding:40px;">Loading...</div>');
            modal.css("display", "flex");
        }

        $.ajax({
            url: smcData.ajaxurl,
            type: "POST",
            data: ajaxData,
            success: function (res) {
                if (res.success) {
                    const { title, content, terms, thumbnail } = res.data;
                    let contentHTML = `<h2 class="home-content-sec" style="color:#e0e0e0;">${title}</h2>`;
                    contentHTML += `<div style="color:#e0e0e0;">${content}</div>`;
                    
                    if (shouldReplaceContent) {
                        exhibitionContent.html(contentHTML);
                        
                        if (res.data.sub_items) {
                            createSubItemsGrid(res.data.sub_items, title, res.data.chapters_label || 'Chapters', exhibitionContent);
                        }
                                                
                        $('html, body').animate({
                            scrollTop: exhibitionContent.offset().top - 100
                        }, 500);
                    } else {
                        modalBody.html(contentHTML);
                    }
                } else {
                    const errorMsg = '<div style="text-align:center; padding:40px; color:#e0e0e0;">載入失敗，請稍後再試。</div>';
                    if (shouldReplaceContent) {
                        exhibitionContent.html(errorMsg);
                    } else {
                        modalBody.html(errorMsg);
                    }
                }
            },
            error: function () {
                const errorMsg = '<div style="text-align:center; padding:40px; color:#e0e0e0;">發生錯誤，請稍後再試。</div>';
                if (shouldReplaceContent) {
                    exhibitionContent.html(errorMsg);
                } else {
                    modalBody.html(errorMsg);
                }
            }
        });
    };

    const hideModal = () => {
        modal.css("display", "none");
    };

    // ===========================================
    // 6. 側邊面板與分類過濾系統
    // ===========================================
    const togglePanel = (panel) => {
        if (panel.hasClass("open")) {
            panel.removeClass("open");
            $('body').removeClass('smc-panel-open');
        } else {
            $(".smc-nav-panel.open").removeClass("open");
            $('body').removeClass('smc-panel-open');
            
            panel.addClass("open");
            $('body').addClass('smc-panel-open');
        }
    };

    const closeAllPanels = () => {
        $(".smc-nav-panel.open").removeClass("open");
        $('body').removeClass('smc-panel-open');
    };

    const filterCardsByCategory = (categorySlug) => {
        if (!categorySlug) {
            cards.show();
            return;
        }

        cards.hide();
        const filteredCards = $(`.smc-card.term-${categorySlug}`);
        filteredCards.show();
        
        if (!isGridMode && isDesktop()) {
            setTimeout(() => {
                generateRandomPositions();
                applyRandomPositions();
            }, 100);
        }
    };

    // ===========================================
    // 7. 語言切換系統
    // ===========================================
    const handleLanguageSwitch = () => {
        if (typeof smcData.currentLang === 'undefined') {
            console.log('Polylang not detected, using fallback');
            const currentUrl = window.location.href;
            if (currentUrl.includes('/en/')) {
                window.location.href = currentUrl.replace('/en/', '/');
            } else {
                window.location.href = currentUrl.replace(smcData.homeUrl, smcData.homeUrl + 'en/');
            }
            return;
        }

        $.ajax({
            url: smcData.ajaxurl,
            type: "POST",
            data: { 
                action: "get_language_switch_url",
                nonce: smcData.nonce
            },
            success: function (res) {
                if (res.success && res.data.url) {
                    window.location.href = res.data.url;
                } else {
                    console.log('Language switch failed');
                }
            },
            error: function () {
                console.log('Language switch error');
            }
        });
    };

    // ===========================================
    // 8. 事件監聽器註冊
    // ===========================================
    
    // 首頁Tab點擊事件
    if (homeTab.length) {
        homeTab.on("click", function (e) {
            e.preventDefault();
            e.stopPropagation();
            showHomeContent();
        });
    }

    // 佈局切換按鈕
    if (layoutToggleBtn.length && isDesktop()) {
        layoutToggleBtn.on("click", function (e) {
            e.preventDefault();
            e.stopPropagation();
            toggleLayout();
        });
    }

    // 卡片點擊事件
    $(document).on("click", ".smc-popup-trigger", function (e) {
        e.preventDefault();
        e.stopPropagation();
        
        if (dragState.isDragging) {
            return;
        }
        
        const postId = $(this).data("post-id");
        const categoryId = $(this).data("category-id");
        
        const isFromSidePanelSubLink = $(this).hasClass('smc-sub-project-link') || 
                                       $(this).closest('.smc-sub-projects').length > 0 ||
                                       $(this).closest('.smc-sub-sub-projects').length > 0;
        
        const forceModal = isFromSidePanelSubLink;
        
        showModal(postId, categoryId, forceModal);
    });

    // Modal關閉按鈕
    if (modalClose.length) {
        modalClose.on("click", hideModal);
    }

    // 點擊Modal背景關閉
    modal.on("click", function (e) {
        if ($(e.target).is(modal)) {
            hideModal();
        }
    });

    // Projects側邊欄
    if (projectsTab.length && projectsPanel.length) {
        projectsTab.on("click", function (e) {
            e.preventDefault();
            e.stopPropagation();
            togglePanel(projectsPanel);
        });
        
        if (projectsClose.length) {
            projectsClose.on("click", function (e) {
                e.preventDefault();
                e.stopPropagation();
                projectsPanel.removeClass("open");
                $('body').removeClass('smc-panel-open');
            });
        }
    }

    // 分類過濾
    $(document).on("click", ".smc-category-filter", function (e) {
        e.preventDefault();
        e.stopPropagation();
        
        const categorySlug = $(this).data("category-slug");
        filterCardsByCategory(categorySlug);
        
        closeAllPanels();
    });

    // 語言切換
    if (languageTab.length) {
        languageTab.on("click", function (e) {
            e.preventDefault();
            e.stopPropagation();
            handleLanguageSwitch();
        });
    }

    // 點擊空白處關閉面板
    $(document).on("click", function (e) {
        const isPanel = $(e.target).closest(".smc-nav-panel").length;
        const isTab = $(e.target).closest(".smc-nav-tab").length;
        const isModal = $(e.target).closest(".smc-modal").length;
        
        if (!isPanel && !isTab && !isModal) {
            closeAllPanels();
        }
    });

    // ESC鍵關閉
    $(document).on("keydown", function (e) {
        if (e.key === "Escape") {
            hideModal();
            closeAllPanels();
        }
    });

    // ===========================================
    // 9. 初始化與響應式處理
    // ===========================================
    
    const initializeLayout = () => {
        // 確保初始狀態：顯示首頁內容，隱藏展覽內容
        if (homeContent.length && exhibitionContent.length) {
            homeContent.show();
            exhibitionContent.hide();
            console.log('Initial state: home content visible, exhibition content hidden');
        }

        if (isMobile()) {
            cardContainer.addClass("smc-grid-mode");
            cards.css({
                'opacity': '1',
                'visibility': 'visible',
                'transition': 'none'
            });
            disableCardDragging();
        } else if (isDesktop() && !isGridMode) {
            generateRandomPositions();
            applyRandomPositions();
            enableCardDragging();
        }
    };

    const handleResize = () => {
        if (isDesktop() && !isGridMode) {
            setTimeout(() => {
                generateRandomPositions();
                applyRandomPositions();
                enableCardDragging();
            }, 300);
        } else {
            disableCardDragging();
            cardContainer.addClass("smc-grid-mode");
            if (isMobile()) {
                cards.css({
                    'opacity': '1',
                    'visibility': 'visible',
                    'transition': 'none'
                });
            }
        }
    };

    initializeLayout();

    let resizeTimeout;
    $(window).on("resize", function () {
        clearTimeout(resizeTimeout);
        resizeTimeout = setTimeout(handleResize, 100);
    });

    $(window).on("load", function () {
        setTimeout(initializeLayout, 50);
    });

    // ===========================================
    // 10. 性能優化與額外功能
    // ===========================================
    
    const debounce = (func, wait) => {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func(...args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    };

    const preloadImages = () => {
        cards.each(function() {
            const img = $(this).find('img');
            if (img.length) {
                const src = img.attr('src');
                if (src) {
                    const preloadImg = new Image();
                    preloadImg.src = src;
                }
            }
        });
    };

    // 鍵盤快捷鍵
    $(document).on("keydown", function (e) {
        // G鍵：切換佈局
        if (e.key.toLowerCase() === "g" && isDesktop() && !$(e.target).is('input, textarea')) {
            e.preventDefault();
            toggleLayout();
        }
        
        // P鍵：打開Projects面板
        if (e.key.toLowerCase() === "p" && !$(e.target).is('input, textarea')) {
            e.preventDefault();
            if (projectsPanel.length) {
                togglePanel(projectsPanel);
            }
        }

        // H鍵：返回首頁
        if (e.key.toLowerCase() === "h" && !$(e.target).is('input, textarea')) {
            e.preventDefault();
            showHomeContent();
        }
    });

    // 手機版觸控手勢
    if (isMobile()) {
        let touchStartX = 0;
        let touchStartY = 0;
        
        $(document).on('touchstart', function(e) {
            touchStartX = e.originalEvent.touches[0].clientX;
            touchStartY = e.originalEvent.touches[0].clientY;
        });
        
        $(document).on('touchend', function(e) {
            const touchEndX = e.originalEvent.changedTouches[0].clientX;
            const touchEndY = e.originalEvent.changedTouches[0].clientY;
            const deltaX = touchEndX - touchStartX;
            const deltaY = touchEndY - touchStartY;
            
            // 從右邊緣向左滑：打開Projects面板
            if (touchStartX > window.innerWidth - 50 && deltaX < -100 && Math.abs(deltaY) < 100) {
                if (projectsPanel.length) {
                    togglePanel(projectsPanel);
                }
            }
            
            // 從左邊緣向右滑：關閉面板
            if (touchStartX < 50 && deltaX > 100 && Math.abs(deltaY) < 100) {
                closeAllPanels();
            }
        });
    }

    setTimeout(preloadImages, 500);

    // ===========================================
    // 11. Debug資訊與錯誤處理
    // ===========================================
    
    window.addEventListener('error', function(e) {
        console.log('SMC Plugin Error:', e.error);
    });

    $(document).ajaxError(function(event, jqxhr, settings, thrownError) {
        console.log('SMC Ajax Error:', thrownError);
    });

    console.log('SMC Plugin initialized successfully');
    console.log('Desktop mode:', isDesktop());
    console.log('Cards found:', cards.length);
    console.log('Current language:', smcData.currentLang || 'unknown');
    console.log('Available elements:', {
        cardContainer: cardContainer.length > 0,
        layoutToggleBtn: layoutToggleBtn.length > 0,
        projectsTab: projectsTab.length > 0,
        projectsPanel: projectsPanel.length > 0,
        languageTab: languageTab.length > 0,
        homeTab: homeTab.length > 0,
        modal: modal.length > 0,
        homeContent: homeContent.length > 0,
        exhibitionContent: exhibitionContent.length > 0
    });
});