<?php
/**
 * Plugin Name: Soundmeridians Core Functions
 * Description: 提供客製化的短代碼與核心功能，包含文章卡片展示、專案導覽、語言切換與彈出視窗。
 * Version: 2.1.2
 * Author: Your Name
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

// 確保外掛不會與其他外掛產生衝突
if (!class_exists('SMC_Core_Plugin')) {
    
    class SMC_Core_Plugin {
        
        private static $instance = null;
        
        public static function get_instance() {
            if (null === self::$instance) {
                self::$instance = new self();
            }
            return self::$instance;
        }
        
        private function __construct() {
            add_action('plugins_loaded', array($this, 'init'));
        }
        
        public function init() {
            // 檢查必要的WordPress功能
            if (!function_exists('wp_enqueue_scripts')) {
                return;
            }
            
            // 初始化外掛功能
            $this->setup_hooks();
            $this->setup_shortcodes();
        }
        
        private function setup_hooks() {
            add_action('wp_enqueue_scripts', array($this, 'enqueue_assets'));
            add_action('wp_ajax_get_post_content', array($this, 'ajax_get_post_content'));
            add_action('wp_ajax_nopriv_get_post_content', array($this, 'ajax_get_post_content'));
            add_action('wp_ajax_get_category_content', array($this, 'ajax_get_category_content'));
            add_action('wp_ajax_nopriv_get_category_content', array($this, 'ajax_get_category_content'));
            add_action('wp_ajax_get_language_switch_url', array($this, 'ajax_get_language_switch_url'));
            add_action('wp_ajax_nopriv_get_language_switch_url', array($this, 'ajax_get_language_switch_url'));
            add_action('init', array($this, 'enable_rest_for_ohio_portfolio'));
        }
        
        private function setup_shortcodes() {
            add_shortcode('smc_main_tabs', array($this, 'shortcode_main_tabs'));
            add_shortcode('smc_project_nav_panel', array($this, 'shortcode_project_nav_panel'));
            add_shortcode('smc_post_card_grid', array($this, 'shortcode_post_card_grid'));
        }
        
        /**
         * 載入 CSS 與 JavaScript 檔案。
         */
        public function enqueue_assets() {
            // 檢查是否在正確的頁面載入資源
            if (is_admin() || wp_doing_ajax()) {
                return;
            }
            
            // 載入 D3.js 庫 (保留以防其他地方使用)
            wp_enqueue_script(
                'd3-library',
                'https://cdnjs.cloudflare.com/ajax/libs/d3/7.8.5/d3.min.js',
                array(),
                '7.8.5',
                true
            );
            
            // 載入 CSS 樣式
            wp_enqueue_style(
                'smc-core-styles', 
                plugin_dir_url(__FILE__) . 'assets/css/sm-core-styles.css', 
                array(), 
                '2.2.10', 
                'all'
            );

            // 載入 JavaScript 腳本
            wp_enqueue_script(
                'smc-core-scripts', 
                plugin_dir_url(__FILE__) . 'assets/js/sm-core-scripts.js', 
                array('jquery', 'd3-library'),
                '2.3.1', 
                true
            );

            // 傳遞資料給 JavaScript
            wp_localize_script('smc-core-scripts', 'smcData', array(
                'ajaxurl' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('smc_nonce'),
                'currentLang' => function_exists('pll_current_language') ? pll_current_language() : 'zh',
                'homeUrl' => home_url('/'),
            ));
        }

        /**
         * 啟用 Ohio Portfolio 的 REST API，以便前端抓取資料。
         */
        public function enable_rest_for_ohio_portfolio() {
            global $wp_post_types;
            if (post_type_exists('ohio_portfolio')) {
                $wp_post_types['ohio_portfolio']->show_in_rest = true;
            }
        }

        /**
         * 取得特定專案（Portfolio）下的子文章（Posts）。
         */
        private function get_project_children($project_id) {
            $project_slug = get_post_field('post_name', $project_id);
            if (!$project_slug) {
                return [];
            }

            $args = [
                'post_type'      => 'post',
                'posts_per_page' => -1,
                'tax_query'      => [
                    [
                        'taxonomy' => 'category',
                        'field'    => 'slug',
                        'terms'    => $project_slug,
                    ],
                ],
                'orderby'        => 'date',
                'order'          => 'ASC',
            ];
            $children_query = new WP_Query($args);
            return $children_query->posts;
        }

        /**
         * 短代碼 [smc_main_tabs]：生成右側三個Tab按鈕。
         */
        public function shortcode_main_tabs() {
        $current_lang = function_exists('pll_current_language') ? pll_current_language() : 'zh';
        
        if ($current_lang === 'zh') {
            $home_text = '聲經絡';
            $projects_text = '發表展覽';
            $lang_switch_text = 'EN';
            // Grid/Random 使用圖標
            $grid_icon = '<i class="fa fa-th" aria-hidden="true"></i>';
            $random_icon = '<i class="fa fa-random" aria-hidden="true"></i>';
        } else {
            $home_text = 'Sound Meridians';
            $projects_text = 'Projects';
            $lang_switch_text = '中文';
            $grid_icon = '<i class="fa fa-th" aria-hidden="true"></i>';
            $random_icon = '<i class="fa fa-random" aria-hidden="true"></i>';
        }
        
        ob_start();
        ?>
        <div class="smc-nav-tabs-container">
            <div class="smc-nav-tab smc-nav-tab-circle" id="smc-layout-toggle-btn" title="<?php echo $current_lang === 'zh' ? '切換佈局' : 'Toggle Layout'; ?>">
                <span id="smc-toggle-icon"><?php echo $grid_icon; ?></span>
            </div>
            <div class="smc-nav-tab" id="smc-home-tab">
                <span><?php echo esc_html($home_text); ?></span>
            </div>
            <div class="smc-nav-tab" id="projects-tab"><?php echo esc_html($projects_text); ?></div>
            <div class="smc-nav-tab" id="language-tab"><?php echo esc_html($lang_switch_text); ?></div>
        </div>
        <?php
        return ob_get_clean();
    }

        /**
         * 短代碼 [smc_project_nav_panel]：生成分類導覽面板。
         */
        public function shortcode_project_nav_panel() {
            // 取得所有父分類（沒有父級的分類），排除「未分類」，以代稱排序
            $parent_categories = get_categories([
                'hide_empty' => false,
                'parent' => 0,
                'orderby' => 'slug',
                'order' => 'ASC',
                'exclude' => array(1) // 排除ID為1的「未分類」
            ]);

            ob_start();
            ?>
            <div id="project-nav-panel" class="smc-nav-panel">
                <div id="project-nav-close" class="smc-close-btn">&times;</div>
                <div class="smc-panel-content">
                    <?php if (!empty($parent_categories)): ?>
                        <div class="smc-projects-grid">
                            <?php foreach ($parent_categories as $parent_cat): 
                                // 跳過「未分類」（通常名稱為 'Uncategorized' 或 '未分類'）
                                if (in_array($parent_cat->slug, ['uncategorized', '未分類', 'uncategorized-en', 'uncategorized-zh']) || 
                                    strpos($parent_cat->slug, 'uncategorized') !== false) {
                                    continue;
                                }
                                
                                // 取得子分類
                                $child_categories = get_categories([
                                    'hide_empty' => false,
                                    'parent' => $parent_cat->term_id,
                                    'orderby' => 'slug',
                                    'order' => 'ASC'
                                ]);

                                // 取得父分類直接包含的文章
                                $direct_posts = get_posts([
                                    'category' => $parent_cat->term_id,
                                    'posts_per_page' => -1,
                                    'orderby' => 'date',
                                    'order' => 'ASC'
                                ]);
                                ?>
                                <div class="smc-project-card">
                                    <div class="smc-project-content">
                                        <h4 class="smc-project-title">
                                            <a href="#" class="smc-project-link smc-popup-trigger" data-category-id="<?php echo esc_attr($parent_cat->term_id); ?>">
                                                <?php echo esc_html($parent_cat->name); ?>
                                            </a>
                                        </h4>
                                        
                                        <?php if (!empty($child_categories)): ?>
                                            <!-- 如果有子分類，顯示子分類和其文章 -->
                                            <ul class="smc-sub-projects">
                                                <?php foreach ($child_categories as $child_cat): 
                                                    // 取得該子分類下的文章
                                                    $posts_in_category = get_posts([
                                                        'category' => $child_cat->term_id,
                                                        'posts_per_page' => -1,
                                                        'orderby' => 'date',
                                                        'order' => 'ASC'
                                                    ]);
                                                    ?>
                                                    <li>
                                                        <i class="fa fa-circle" aria-hidden="true"></i> <strong><?php echo esc_html($child_cat->name); ?></strong>
                                                        <?php if (!empty($posts_in_category)): ?>
                                                            <ul class="smc-sub-sub-projects">
                                                                <?php foreach ($posts_in_category as $post): ?>
                                                                    <li>
                                                                        <i class="fa fa-circle" aria-hidden="true"></i> <a href="#" class="smc-sub-project-link smc-popup-trigger" data-post-id="<?php echo esc_attr($post->ID); ?>">
                                                                            <?php echo esc_html($post->post_title); ?>
                                                                        </a>
                                                                    </li>
                                                                <?php endforeach; ?>
                                                            </ul>
                                                        <?php endif; ?>
                                                    </li>
                                                <?php endforeach; ?>
                                            </ul>
                                        <?php elseif (!empty($direct_posts)): ?>
                                            <!-- 如果沒有子分類但有直接文章，顯示這些文章 -->
                                            <ul class="smc-sub-sub-projects">
                                                <?php foreach ($direct_posts as $post): ?>
                                                    <li>
                                                        <i class="fa fa-circle" aria-hidden="true"></i> <a href="#" class="smc-sub-project-link smc-popup-trigger" data-post-id="<?php echo esc_attr($post->ID); ?>">
                                                            <?php echo esc_html($post->post_title); ?>
                                                        </a>
                                                    </li>
                                                <?php endforeach; ?>
                                            </ul>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php else: ?>
                        <p>No categories found.</p>
                    <?php endif; ?>
                </div>
            </div>
            <?php
            return ob_get_clean();
        }

        /**
         * 短代碼 [smc_post_card_grid]：生成只顯示父分類卡片的格子。
         */
        public function shortcode_post_card_grid() {
            // 取得所有父分類(第一層分類)
            $parent_categories = get_categories([
                'hide_empty' => false,
                'parent' => 0,
                'orderby' => 'slug',
                'order' => 'ASC',
                'exclude' => array(1) // 排除「未分類」
            ]);

            // 取得當前語言
            $current_lang = function_exists('pll_current_language') ? pll_current_language() : 'zh';
            
            // 根據語言設定標籤文字
            $exhibition_text = ($current_lang === 'zh') ? '發表展覽' : 'Project';

            ob_start();

            // 彈出視窗的 HTML 結構
            ?>
            <div id="smc-card-modal" class="smc-modal">
                <div class="smc-modal-content">
                    <button id="smc-modal-close" class="smc-close-btn">&times;</button>
                    <div id="smc-modal-body"></div>
                </div>
            </div>

            <div id="smc-card-container">
                <?php
                if (!empty($parent_categories)) {
                    foreach ($parent_categories as $category) {
                        // 跳過「未分類」（包含中英文及所有語言版本）
                        if (in_array($category->slug, ['uncategorized', '未分類', 'uncategorized-en', 'uncategorized-zh']) || 
                            strpos($category->slug, 'uncategorized') !== false) {
                            continue;
                        }
                        
                        // 取得分類的縮圖：從 ohio_portfolio 取得
                        $thumbnail_url = '';
                        
                        // 方法1: 用 slug 匹配
                        $portfolio_post = get_posts([
                            'post_type' => 'ohio_portfolio',
                            'name' => $category->slug,
                            'posts_per_page' => 1,
                            'post_status' => 'publish'
                        ]);
                        
                        // 除錯：記錄查找結果
                        error_log('Category: ' . $category->name . ' (slug: ' . $category->slug . ')');
                        error_log('Portfolio found: ' . (empty($portfolio_post) ? 'NO' : 'YES'));
                        
                        if (!empty($portfolio_post)) {
                            error_log('Portfolio ID: ' . $portfolio_post[0]->ID);
                            error_log('Portfolio Title: ' . $portfolio_post[0]->post_title);
                            
                            // 嘗試不同的圖片尺寸
                            $thumbnail_url = get_the_post_thumbnail_url($portfolio_post[0]->ID, 'full');
                            
                            if (!$thumbnail_url) {
                                $thumbnail_url = get_the_post_thumbnail_url($portfolio_post[0]->ID, 'large');
                            }
                            if (!$thumbnail_url) {
                                $thumbnail_url = get_the_post_thumbnail_url($portfolio_post[0]->ID, 'medium');
                            }
                            if (!$thumbnail_url) {
                                $thumbnail_url = get_the_post_thumbnail_url($portfolio_post[0]->ID, 'thumbnail');
                            }
                            if (!$thumbnail_url) {
                                // 最後嘗試：直接取得附件 ID
                                $thumbnail_id = get_post_thumbnail_id($portfolio_post[0]->ID);
                                if ($thumbnail_id) {
                                    $thumbnail_url = wp_get_attachment_url($thumbnail_id);
                                    error_log('Got thumbnail from attachment ID: ' . $thumbnail_url);
                                }
                            }
                            
                            error_log('Thumbnail URL: ' . ($thumbnail_url ? $thumbnail_url : 'NONE'));
                        }
                        
                        // 方法2: 如果用 slug 找不到，嘗試用 title 匹配
                        if (empty($portfolio_post)) {
                            error_log('Trying to match by title...');
                            $portfolio_post = get_posts([
                                'post_type' => 'ohio_portfolio',
                                'title' => $category->name,
                                'posts_per_page' => 1,
                                'post_status' => 'publish'
                            ]);
                            
                            if (!empty($portfolio_post)) {
                                error_log('Found by title! ID: ' . $portfolio_post[0]->ID);
                                $thumbnail_id = get_post_thumbnail_id($portfolio_post[0]->ID);
                                if ($thumbnail_id) {
                                    $thumbnail_url = wp_get_attachment_url($thumbnail_id);
                                }
                            }
                        }
                        
                        $term_slug_class = 'term-' . sanitize_html_class($category->slug);
                        ?>
                        <div class="smc-card smc-draggable <?php echo esc_attr($term_slug_class); ?>" data-category-id="<?php echo esc_attr($category->term_id); ?>">
                            <?php if ($thumbnail_url): ?>
                                <img class="smc-card-thumb" src="<?php echo esc_url($thumbnail_url); ?>" alt="<?php echo esc_attr($category->name); ?>" loading="lazy">
                            <?php else: ?>
                                <!-- 除錯資訊：顯示分類 slug -->
                                <div class="smc-card-thumb smc-card-placeholder" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); display: flex; flex-direction: column; align-items: center; justify-content: center; color: white; font-size: 48px; font-weight: 300; height: 200px;">
                                    <div><?php echo esc_html(mb_substr($category->name, 0, 1)); ?></div>
                                    <div style="font-size: 10px; margin-top: 10px; opacity: 0.7;">slug: <?php echo esc_html($category->slug); ?></div>
                                </div>
                            <?php endif; ?>
                            <div class="smc-card-content" style="position: relative;">
                                <h4 class="smc-card-title"><?php echo esc_html($category->name); ?></h4>
                                <div class="smc-card-category"><i class="fa fa-circle" aria-hidden="true"></i>&nbsp;<?php echo esc_html($exhibition_text); ?></div>
                                <div style="position: absolute; bottom: 10px; right: 10px;">
                                    <button class="smc-popup-trigger" data-category-id="<?php echo esc_attr($category->term_id); ?>" style="width: 32px; height: 32px; background: rgba(0, 0, 0, 0.8); border: 0.1px solid rgba(255,255,255,0.7); color: #fff; border-radius: 50%; cursor: pointer; display: flex; align-items: center; justify-content: center; transition: all 0.3s ease;" title="查看內容 / View Content">
                                        <i class="fa fa-eye" aria-hidden="true" style="font-size: 12px;"></i>
                                    </button>
                                </div>
                            </div>
                        </div>
                        <?php
                    }
                } else {
                    echo '<p>沒有找到任何分類。</p>';
                }
                ?>
            </div>
            <?php
            return ob_get_clean();
        }

        /**
         * 處理 Ajax 請求以獲取分類內容。
         */
        public function ajax_get_category_content() {
            // 驗證 nonce
            if (!wp_verify_nonce($_POST['nonce'] ?? '', 'smc_nonce')) {
                wp_send_json_error('Invalid nonce');
                return;
            }

            if (!isset($_POST['category_id']) || !is_numeric($_POST['category_id'])) {
                wp_send_json_error('Invalid category ID');
                return;
            }

            $category_id = intval($_POST['category_id']);
            $category = get_category($category_id);
            
            if (!$category || is_wp_error($category)) {
                wp_send_json_error('Category not found');
                return;
            }

            // 根據分類名稱查找對應的 ohio_portfolio 文章
            $portfolio_post = get_posts([
                'post_type' => 'ohio_portfolio',
                'name' => $category->slug, // 使用分類的 slug 來匹配文章的 slug
                'posts_per_page' => 1,
                'post_status' => 'publish'
            ]);

            // 如果沒有找到，嘗試用分類名稱查找
            if (empty($portfolio_post)) {
                $portfolio_post = get_posts([
                    'post_type' => 'ohio_portfolio',
                    'title' => $category->name, // 用分類名稱匹配文章標題
                    'posts_per_page' => 1,
                    'post_status' => 'publish'
                ]);
            }

            $content = '';
            $thumbnail_url = '';
            
            if (!empty($portfolio_post)) {
                $post = $portfolio_post[0];
                $content = apply_filters('the_content', $post->post_content);
                $thumbnail_url = get_the_post_thumbnail_url($post->ID, 'large');
            } else {
                // 如果沒有對應的 portfolio，顯示該分類下的文章列表
                $posts_in_category = get_posts([
                    'category' => $category_id,
                    'posts_per_page' => -1,
                    'post_status' => 'publish'
                ]);
                
                if (!empty($posts_in_category)) {
                    $content = '<h3>' . $category->name . ' 項目列表</h3><ul>';
                    foreach ($posts_in_category as $post) {
                        $content .= '<li><a href="' . get_permalink($post->ID) . '">' . $post->post_title . '</a></li>';
                    }
                    $content .= '</ul>';
                } else {
                    $content = $category->description ? $category->description : '找不到對應的內容。';
                }
            }

            // 獲取分類的子項目數據（用於Grid顯示）
            $sub_items = $this->get_category_sub_items($category_id);
            
            // 多語言標籤
            $current_lang = function_exists('pll_current_language') ? pll_current_language() : 'zh';
            $category_label = ($current_lang === 'zh') ? '發表展覽' : 'Exhibition';
            $chapters_label = ($current_lang === 'zh') ? '章節' : 'Chapters';

            wp_send_json_success(array(
                'title' => $category->name,
                'content' => $content,
                'terms' => $category_label,
                'thumbnail' => $thumbnail_url,
                'sub_items' => $sub_items,
                'chapters_label' => $chapters_label
            ));
        }

        /**
         * 獲取分類的子項目數據（用於Grid顯示）
         */
        private function get_category_sub_items($parent_category_id) {
            $parent_category = get_category($parent_category_id);
            if (!$parent_category) {
                return array();
            }

            $sub_items = array();

            // 獲取子分類
            $child_categories = get_categories([
                'parent' => $parent_category_id,
                'hide_empty' => false,
                'orderby' => 'slug',
                'order' => 'ASC'
            ]);

            // 如果沒有子分類，直接返回該分類下的文章
            if (empty($child_categories)) {
                $direct_posts = get_posts([
                    'category' => $parent_category_id,
                    'posts_per_page' => -1,
                    'post_status' => 'publish',
                    'orderby' => 'date',
                    'order' => 'ASC'
                ]);

                foreach ($direct_posts as $post) {
                    $post_thumbnail = get_the_post_thumbnail_url($post->ID, 'medium');
                    $sub_items[] = array(
                        'id' => $post->ID,
                        'title' => $post->post_title,
                        'type' => 'post',
                        'thumbnail' => $post_thumbnail,
                        'has_children' => false,
                        'children' => array()
                    );
                }
                return $sub_items;
            }

            foreach ($child_categories as $child_cat) {
                // 檢查是否有對應的文章（slug相同）
                $corresponding_post = get_posts([
                    'name' => $child_cat->slug,
                    'post_type' => 'post',
                    'posts_per_page' => 1,
                    'post_status' => 'publish'
                ]);

                // 獲取縮圖
                $thumbnail_url = '';
                if (!empty($corresponding_post)) {
                    $thumbnail_url = get_the_post_thumbnail_url($corresponding_post[0]->ID, 'medium');
                }

                // 如果沒有對應文章的縮圖，從該子分類下的第一篇文章取得
                if (!$thumbnail_url) {
                    $first_post = get_posts([
                        'category' => $child_cat->term_id,
                        'posts_per_page' => 1,
                        'post_status' => 'publish'
                    ]);
                    if (!empty($first_post)) {
                        $thumbnail_url = get_the_post_thumbnail_url($first_post[0]->ID, 'medium');
                    }
                }

                // 獲取該子分類下的所有文章
                $posts = get_posts([
                    'category' => $child_cat->term_id,
                    'posts_per_page' => -1,
                    'post_status' => 'publish',
                    'orderby' => 'date',
                    'order' => 'ASC'
                ]);

                $sub_posts = array();
                foreach ($posts as $post) {
                    // 排除與分類同名的文章
                    if ($post->post_name !== $child_cat->slug) {
                        $post_thumbnail = get_the_post_thumbnail_url($post->ID, 'medium');
                        $sub_posts[] = array(
                            'id' => $post->ID,
                            'title' => $post->post_title,
                            'type' => 'post',
                            'thumbnail' => $post_thumbnail
                        );
                    }
                }

                // 決定項目類型
                if (!empty($corresponding_post)) {
                    $item_type = 'post';
                    $item_id = $corresponding_post[0]->ID;
                } else {
                    $item_type = 'category';
                    $item_id = $child_cat->term_id;
                }

                $sub_items[] = array(
                    'id' => $item_id,
                    'title' => $child_cat->name,
                    'type' => $item_type,
                    'thumbnail' => $thumbnail_url,
                    'has_children' => !empty($sub_posts),
                    'children' => $sub_posts
                );
            }

            return $sub_items;
        }

        /**
         * 處理 Ajax 請求以獲取文章內容。
         */
        public function ajax_get_post_content() {
            // 驗證 nonce
            if (!wp_verify_nonce($_POST['nonce'] ?? '', 'smc_nonce')) {
                wp_send_json_error('Invalid nonce');
                return;
            }

            if (!isset($_POST['post_id']) || !is_numeric($_POST['post_id'])) {
                wp_send_json_error('Invalid post ID');
                return;
            }

            $post_id = intval($_POST['post_id']);
            $post = get_post($post_id);
            
            if (!$post || $post->post_status !== 'publish') {
                wp_send_json_error('Post not found or not published');
                return;
            }

            $terms = get_the_terms($post_id, 'category');
            $term_names = '';
            if ($terms && !is_wp_error($terms)) {
                $term_names = implode(', ', wp_list_pluck($terms, 'name'));
            }

            $content = apply_filters('the_content', $post->post_content);
            $thumbnail_url = get_the_post_thumbnail_url($post_id, 'large');

            wp_send_json_success(array(
                'title' => get_the_title($post_id),
                'content' => $content,
                'terms' => $term_names,
                'thumbnail' => $thumbnail_url
            ));
        }

        /**
         * 處理語言切換的Ajax請求
         */
        public function ajax_get_language_switch_url() {
            // 驗證 nonce
            if (!wp_verify_nonce($_POST['nonce'] ?? '', 'smc_nonce')) {
                wp_send_json_error('Invalid nonce');
                return;
            }

            if (!function_exists('pll_current_language') || !function_exists('pll_the_languages')) {
                wp_send_json_error('Polylang not available');
                return;
            }

            $current_lang = pll_current_language();
            $post_id = get_the_ID();
            
            // 取得其他語言版本的URL
            $languages = pll_the_languages(array('raw' => 1));
            $switch_url = home_url('/');
            
            foreach ($languages as $lang) {
                if ($lang['slug'] !== $current_lang) {
                    if ($post_id && function_exists('pll_get_post')) {
                        $translated_post_id = pll_get_post($post_id, $lang['slug']);
                        if ($translated_post_id) {
                            $switch_url = get_permalink($translated_post_id);
                        } else {
                            $switch_url = $lang['url'];
                        }
                    } else {
                        $switch_url = $lang['url'];
                    }
                    break;
                }
            }

            wp_send_json_success(array('url' => $switch_url));
        }
    }

    // 初始化外掛
    SMC_Core_Plugin::get_instance();
}