console.clear();

const canvas = document.getElementById('canvas');
const videoDom = document.getElementById('video');
const scene = new THREE.Scene();
const renderer = new THREE.WebGLRenderer({canvas: canvas, antialias: true, alpha: true});
const camera = new THREE.PerspectiveCamera(45, canvas.clientWidth / canvas.clientWidth, 1, 1000);
const mouse = new THREE.Vector2(0.15, 0.3);

// 載入聲經絡背景圖片
const dogTexture = new THREE.TextureLoader().load('https://dev.8f-2.cc/soundmeridians01/wp-content/uploads/2025/07/SoundMeridians-0626_bg2.jpg');

var quad = new THREE.Mesh(
  new THREE.PlaneGeometry(2, 2),
  new THREE.ShaderMaterial({
    vertexShader: document.getElementById('vertex-shader').textContent,
    fragmentShader: document.getElementById('fragment-shader').textContent,
    depthWrite: false,
    depthTest: false,
    uniforms: {
      dog: {
        type: "t",
        value: dogTexture
      },
      delta: {
        value: 0
      },
      mouse: {
        value: mouse
      },
      filter: {
        value: false
      },
      speed: {
        value: 0.015  // 極緩慢的速度，像聲波在經絡中流動
      },
      distortionStrength: {
        value: 0.35  // 降低扭曲強度，保持圖片的完整性
      },
      waveFrequency: {
        value: 0.8   // 降低波動頻率，更柔和的律動
      }
    }
  })
);
scene.add(quad);

// 響應式調整
function onResize () {
  renderer.setSize(canvas.clientWidth, canvas.clientHeight, false);
  camera.aspect = canvas.clientWidth / canvas.clientHeight;
  camera.updateProjectionMatrix();
}

// 滑鼠移動事件 - 非常緩慢的回應，像經絡中的氣血流動
function onMouseMove(e) {
  const x = (e.clientX / (window.innerWidth * 0.5)) - 1;
  const y = -1 * (e.clientY / (window.innerHeight * 0.5)) + 1;
  updateMouse(x, y);
}

function onTouchMove(e) {
  const x = (e.touches[0].clientX / (window.innerWidth * 0.5)) - 1;
  const y = -1 * (e.touches[0].clientY / (window.innerHeight * 0.5)) + 1;
  updateMouse(x, y);
}

function updateMouse(x, y) {
  // 極度緩慢的緩動，5秒才完成移動
  TweenMax.to(mouse, 5, {
    x: x * 0.5,  // 降低滑鼠影響到 50%
    y: y * 0.5,
    ease: Power1.easeOut,  // 更柔和的緩動曲線
    onUpdate: function () {
      quad.material.uniforms.mouse.value = mouse;
    }
  });
}

// 聲波經絡的自動流動效果 - 模擬聲音在不同文化脈絡中的流動
let autoTime = 0;
let pulsePhase = 0;

function meridianFlow() {
  autoTime += 0.0003;      // 極緩慢的時間推進
  pulsePhase += 0.0008;    // 脈動相位
  
  // 主要流動：模擬聲波的傳播路徑
  const flowX = Math.sin(autoTime * 0.8) * 0.12;
  const flowY = Math.cos(autoTime * 0.5) * 0.12;
  
  // 次要脈動：模擬不同聲響文化的共鳴
  const pulseX = Math.sin(pulsePhase * 1.3) * 0.08;
  const pulseY = Math.cos(pulsePhase * 0.9) * 0.08;
  
  // 微妙的旋轉效果：模擬聲音的環繞感
  const rotateX = Math.sin(autoTime * 0.3) * 0.05;
  const rotateY = Math.cos(autoTime * 0.4) * 0.05;
  
  // 混合所有運動
  const targetX = mouse.x * 0.4 + flowX + pulseX + rotateX;
  const targetY = mouse.y * 0.4 + flowY + pulseY + rotateY;
  
  // 非常緩慢地插值到目標位置
  quad.material.uniforms.mouse.value.x += (targetX - quad.material.uniforms.mouse.value.x) * 0.005;
  quad.material.uniforms.mouse.value.y += (targetY - quad.material.uniforms.mouse.value.y) * 0.005;
}

// 週期性的能量波動 - 模擬聲音的起伏
let energyWave = 0;
function energyPulse() {
  energyWave += 0.0002;
  const pulse = Math.sin(energyWave) * 0.5 + 0.5;  // 0 到 1 之間振盪
  
  // 微調速度，產生呼吸般的律動感
  const breathSpeed = 0.015 + pulse * 0.005;
  quad.material.uniforms.speed.value = breathSpeed;
}

// 渲染循環
function render(a) {
  requestAnimationFrame(render);
  
  // 極度緩慢的時間進度，保持圖片的可辨識性
  quad.material.uniforms.delta.value = a * 0.0002;
  
  // 執行經絡流動效果
  meridianFlow();
  
  // 執行能量脈動
  energyPulse();

  renderer.render(scene, camera);
}

// 初始化
onResize();
window.addEventListener('resize', onResize);
window.addEventListener('mousemove', onMouseMove);
window.addEventListener('touchmove', onTouchMove);

// 滑鼠離開時，緩慢回到初始狀態
window.addEventListener('mouseleave', function() {
  TweenMax.to(mouse, 6, {
    x: 0.15, 
    y: 0.3,
    ease: Power1.easeInOut
  });
});

// 頁面可見性處理
let isVisible = true;
document.addEventListener('visibilitychange', function() {
  if (document.hidden) {
    isVisible = false;
    console.log('聲經絡背景已暫停');
  } else {
    isVisible = true;
    console.log('聲經絡背景已恢復');
  }
});

// 啟動渲染
requestAnimationFrame(render);

console.log('聲經絡 - 聲響文化製圖 背景效果已啟動');