<?php
/**
 * Plugin Name: Poetry Screensaver Effect
 * Description: 詩歌浮現螢幕保護程式效果，支援多語系KTV歌詞式顯示，Dark Mode優化
 * Version: 1.1
 * Author: Custom Plugin
 */

// 防止直接訪問
if (!defined('ABSPATH')) {
    exit;
}

class PoetryScreensaverPlugin {
    
    private $poems = array(
        'zh' => array(
            '抹除的實例',
            '淡出的發生',
            '空缺的片刻',
            '移除的事件',
            '片段的瓦解',
            '不可見的顯現',
            '隱匿的行為',
            '抽離的過程',
            '鬆解、崩塌、消失',
            '取消生成、拒絕成為、反轉想像'
        ),
        'en' => array(
            'Instances of erasure',
            'Occurrences of fading',
            'Moments of absence',
            'Incidents of removal',
            'Episodes of dissolution',
            'Manifestations of invisibility',
            'Acts of concealment',
            'Processes of withdrawal',
            'To decompress, to collapse, to disappear',
            'Unmaking, unbecoming, undoing'
        ),
        'es' => array(
            'Casos de borrado',
            'Episodios de desvanecimiento',
            'Momentos de ausencia',
            'Incidentes de eliminación',
            'Episodios de disolución',
            'Manifestaciones de invisibilidad',
            'Actos de ocultación',
            'Procesos de retirada',
            'Descomprimir, colapsar, ocultar',
            'Deshacer, desmoronar, desvanecer'
        )
    );

    public function __construct() {
        add_action('init', array($this, 'init'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('wp_footer', array($this, 'add_poetry_html'));
        add_action('admin_menu', array($this, 'admin_menu'));
        add_action('wp_ajax_get_poetry_data', array($this, 'get_poetry_data'));
        add_action('wp_ajax_nopriv_get_poetry_data', array($this, 'get_poetry_data'));
        
        // 設定預設值
        register_activation_hook(__FILE__, array($this, 'set_default_options'));
    }

    public function set_default_options() {
        add_option('poetry_screensaver_enabled', '1');
        add_option('poetry_screensaver_speed', '15');
        add_option('poetry_screensaver_interval', '3');
        add_option('poetry_screensaver_font_size_rem', '1.5');
        add_option('poetry_screensaver_idle_time', '60');
        add_option('poetry_screensaver_mode', '1');
        add_option('poetry_screensaver_overlay_opacity', '0.3');
    }

    public function init() {
        // 註冊設定
        register_setting('poetry_screensaver_settings', 'poetry_screensaver_enabled');
        register_setting('poetry_screensaver_settings', 'poetry_screensaver_speed');
        register_setting('poetry_screensaver_settings', 'poetry_screensaver_interval');
        register_setting('poetry_screensaver_settings', 'poetry_screensaver_font_size_rem');
        register_setting('poetry_screensaver_settings', 'poetry_screensaver_idle_time');
        register_setting('poetry_screensaver_settings', 'poetry_screensaver_mode');
        register_setting('poetry_screensaver_settings', 'poetry_screensaver_overlay_opacity');
    }

    public function enqueue_scripts() {
        wp_enqueue_style('poetry-screensaver-fonts', 'https://fonts.googleapis.com/css2?family=Playfair+Display:ital,wght@0,300;0,400;1,300;1,400&family=Crimson+Text:ital,wght@0,400;0,600;1,400;1,600&family=Noto+Serif+TC:wght@300;400&display=swap', array(), null);
        
        wp_enqueue_style('poetry-screensaver-style', plugins_url('style.css', __FILE__), array(), '1.5');
        wp_enqueue_script('poetry-screensaver-script', plugins_url('script.js', __FILE__), array('jquery'), '1.1', true);
        
        // 傳遞資料到JavaScript
        wp_localize_script('poetry-screensaver-script', 'poetryScreensaverData', array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('poetry_screensaver_nonce'),
            'settings' => array(
                'enabled' => get_option('poetry_screensaver_enabled', '1'),
                'speed' => get_option('poetry_screensaver_speed', '15'),
                'interval' => get_option('poetry_screensaver_interval', '3'),
                'fontSizeRem' => get_option('poetry_screensaver_font_size_rem', '1.5'),
                'idleTime' => get_option('poetry_screensaver_idle_time', '60'),
                'screensaverMode' => get_option('poetry_screensaver_mode', '1'),
                'overlayOpacity' => get_option('poetry_screensaver_overlay_opacity', '0.3')
            ),
            'debug' => defined('WP_DEBUG') && WP_DEBUG
        ));
    }

    public function add_poetry_html() {
        echo '<div id="poetry-screensaver-container" style="display: none;"></div>';
        echo '<div id="poetry-screensaver-overlay" style="display: none;" class="jf-sunday"></div>';
    }

    public function get_poetry_data() {
        // 檢查nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'poetry_screensaver_nonce')) {
            wp_send_json_error('Security check failed');
            return;
        }

        // 獲取當前語言
        $current_lang = 'en';
        
        if (function_exists('pll_current_language')) {
            $lang = pll_current_language();
            if ($lang) $current_lang = $lang;
        } elseif (function_exists('ICL_LANGUAGE_CODE') && defined('ICL_LANGUAGE_CODE')) {
            $current_lang = ICL_LANGUAGE_CODE;
        }

        // 語言映射
        $lang_map = array(
            'zh-tw' => 'zh', 'zh-cn' => 'zh', 'zh-hant' => 'zh', 'zh-hans' => 'zh', 'zh' => 'zh',
            'en' => 'en', 'en-us' => 'en', 'en-gb' => 'en',
            'es' => 'es', 'es-es' => 'es'
        );

        $mapped_lang = isset($lang_map[$current_lang]) ? $lang_map[$current_lang] : 'en';
        
        if (!isset($this->poems[$mapped_lang])) {
            $mapped_lang = 'en';
        }

        $poems = $this->poems[$mapped_lang];

        wp_send_json_success(array(
            'poems' => $poems,
            'language' => $mapped_lang,
            'count' => count($poems)
        ));
    }

    public function admin_menu() {
        add_options_page(
            '詩歌螢幕保護程式',
            '詩歌螢幕保護程式',
            'manage_options',
            'poetry-screensaver-settings',
            array($this, 'admin_page')
        );
    }

    public function admin_page() {
        $message = '';
        
        if (isset($_POST['submit']) && wp_verify_nonce($_POST['_wpnonce'], 'poetry_screensaver_settings_nonce')) {
            update_option('poetry_screensaver_enabled', isset($_POST['poetry_screensaver_enabled']) ? '1' : '0');
            update_option('poetry_screensaver_speed', intval($_POST['poetry_screensaver_speed']));
            update_option('poetry_screensaver_interval', intval($_POST['poetry_screensaver_interval']));
            update_option('poetry_screensaver_font_size_rem', floatval($_POST['poetry_screensaver_font_size_rem']));
            update_option('poetry_screensaver_idle_time', intval($_POST['poetry_screensaver_idle_time']));
            update_option('poetry_screensaver_mode', isset($_POST['poetry_screensaver_mode']) ? '1' : '0');
            update_option('poetry_screensaver_overlay_opacity', floatval($_POST['poetry_screensaver_overlay_opacity']));
            
            $message = '<div class="notice notice-success"><p>設定已成功儲存！</p></div>';
        }

        $enabled = get_option('poetry_screensaver_enabled', '1');
        $speed = get_option('poetry_screensaver_speed', '15');
        $interval = get_option('poetry_screensaver_interval', '3');
        $font_size_rem = get_option('poetry_screensaver_font_size_rem', '1.5');
        $idle_time = get_option('poetry_screensaver_idle_time', '60');
        $screensaver_mode = get_option('poetry_screensaver_mode', '1');
        $overlay_opacity = get_option('poetry_screensaver_overlay_opacity', '0.3');
        ?>
        <div class="wrap">
            <h1>詩歌螢幕保護程式設定</h1>
            <?php echo $message; ?>
            
            <form method="post" action="">
                <?php wp_nonce_field('poetry_screensaver_settings_nonce'); ?>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">啟用效果</th>
                        <td>
                            <label>
                                <input type="checkbox" name="poetry_screensaver_enabled" value="1" <?php checked($enabled, '1'); ?>>
                                啟用詩歌螢幕保護程式效果（全站）
                            </label>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">螢幕保護程式模式</th>
                        <td>
                            <label>
                                <input type="checkbox" name="poetry_screensaver_mode" value="1" <?php checked($screensaver_mode, '1'); ?>>
                                啟用螢幕保護程式模式（閒置時自動播放）
                            </label>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">閒置時間</th>
                        <td>
                            <input type="number" name="poetry_screensaver_idle_time" value="<?php echo esc_attr($idle_time); ?>" min="10" max="300" step="10" style="width: 80px;">
                            秒
                            <p class="description">使用者無操作多久後開始播放詩句</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">字體大小</th>
                        <td>
                            <input type="number" name="poetry_screensaver_font_size_rem" value="<?php echo esc_attr($font_size_rem); ?>" min="0.8" max="4.0" step="0.1" style="width: 80px;">
                            rem
                            <p class="description">詩句文字大小（建議 1.0-3.0，1.5rem ≈ 24px）</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">背景遮罩透明度</th>
                        <td>
                            <input type="range" name="poetry_screensaver_overlay_opacity" value="<?php echo esc_attr($overlay_opacity); ?>" min="0" max="0.8" step="0.1" oninput="document.getElementById('opacity-display').textContent = this.value">
                            <span id="opacity-display"><?php echo esc_html($overlay_opacity); ?></span>
                            <p class="description">螢幕保護程式時的背景遮罩透明度</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">動畫速度</th>
                        <td>
                            <input type="number" name="poetry_screensaver_speed" value="<?php echo esc_attr($speed); ?>" min="5" max="30" step="1" style="width: 80px;">
                            秒
                            <p class="description">每行詩句移動的時間</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">詩句間隔</th>
                        <td>
                            <input type="number" name="poetry_screensaver_interval" value="<?php echo esc_attr($interval); ?>" min="1" max="10" step="1" style="width: 80px;">
                            秒
                            <p class="description">每行詩句出現的間隔時間</p>
                        </td>
                    </tr>
                </table>
                
                <div style="background: #f0f0f1; padding: 15px; border-radius: 5px; margin: 20px 0;">
                    <h3>🧪 測試快捷鍵</h3>
                    <ul style="margin: 10px 0;">
                        <li><code>Ctrl + Shift + T</code> - 立即觸發螢幕保護程式</li>
                        <li><code>Ctrl + Shift + S</code> - 停止螢幕保護程式</li>
                        <li><code>Ctrl + Shift + P</code> - 暫停/播放詩句</li>
                    </ul>
                    <p style="margin: 10px 0; color: #666;">在瀏覽器控制台輸入 <code>PoetryScreensaverControl.debug()</code> 可查看詳細狀態</p>
                </div>
                
                <?php submit_button('儲存設定'); ?>
            </form>
        </div>
        <?php
    }
}

// 初始化外掛
new PoetryScreensaverPlugin();
