// Poetry Screensaver JavaScript - Simplified Lightweight Version
(function($) {
    'use strict';

    class PoetryScreensaver {
        constructor() {
            this.container = $('#poetry-screensaver-container');
            this.overlay = $('#poetry-screensaver-overlay');
            this.poems = [];
            this.currentIndex = 0;
            this.isRunning = false;
            this.isScreensaverActive = false;
            this.language = 'en';
            this.idleTimer = null;
            this.lastActivity = Date.now();
            this.sequenceTimer = null;
            
            // 從WordPress傳來的設定
            this.settings = {
                enabled: poetryScreensaverData.settings.enabled === '1',
                speed: parseInt(poetryScreensaverData.settings.speed) || 15,
                interval: parseInt(poetryScreensaverData.settings.interval) || 3,
                fontSizeRem: parseFloat(poetryScreensaverData.settings.fontSizeRem) || 1.5,
                idleTime: parseInt(poetryScreensaverData.settings.idleTime) || 60,
                screensaverMode: poetryScreensaverData.settings.screensaverMode === '1',
                overlayOpacity: parseFloat(poetryScreensaverData.settings.overlayOpacity) || 0.3
            };
            
            this.debug = poetryScreensaverData.debug || false;
            
            this.log('Initializing Lightweight Poetry Screensaver');
            this.init();
        }

        log(...args) {
            if (this.debug || window.poetryScreensaverDebug) {
                console.log('Poetry Screensaver:', ...args);
            }
        }

        init() {
            if (!this.settings.enabled) {
                this.log('Plugin disabled in settings');
                return;
            }
            
            this.setupContainers();
            this.loadPoetryData();
            this.bindEvents();
            
            if (this.settings.screensaverMode) {
                this.startIdleTimer();
            }
            
            this.log('Lightweight version initialized successfully');
        }

        setupContainers() {
            this.container.css('display', 'block');
            this.overlay.css({
                'display': 'block',
                'background-color': `rgba(0, 0, 0, ${this.settings.overlayOpacity})`
            });
        }

        loadPoetryData() {
            this.log('Loading poetry data...');
            
            $.ajax({
                url: poetryScreensaverData.ajaxurl,
                type: 'POST',
                data: {
                    action: 'get_poetry_data',
                    nonce: poetryScreensaverData.nonce
                },
                success: (response) => {
                    if (response.success && response.data.poems && response.data.poems.length > 0) {
                        this.poems = response.data.poems;
                        this.language = response.data.language;
                        this.log(`Loaded ${this.poems.length} poems in ${this.language}`);
                    } else {
                        this.log('Using fallback poems');
                        this.useDefaultPoems();
                    }
                },
                error: () => {
                    this.log('AJAX failed, using fallback poems');
                    this.useDefaultPoems();
                }
            });
        }

        useDefaultPoems() {
            this.poems = [
                'Instances of erasure',
                'Occurrences of fading',
                'Moments of absence',
                'Incidents of removal',
                'Episodes of dissolution',
                'Manifestations of invisibility',
                'Acts of concealment',
                'Processes of withdrawal',
                'To decompress, to collapse, to disappear',
                'Unmaking, unbecoming, undoing'
            ];
            this.language = 'en';
        }

        startIdleTimer() {
            this.resetIdleTimer();
        }

        resetIdleTimer() {
            this.lastActivity = Date.now();
            
            if (this.idleTimer) {
                clearTimeout(this.idleTimer);
                this.idleTimer = null;
            }
            
            if (this.isScreensaverActive) {
                this.stopScreensaver();
            }
            
            this.idleTimer = setTimeout(() => {
                this.startScreensaver();
            }, this.settings.idleTime * 1000);
        }

        startScreensaver() {
            if (this.isScreensaverActive || this.poems.length === 0) {
                return;
            }
            
            this.log('Starting screensaver mode');
            this.isScreensaverActive = true;
            this.currentIndex = 0;
            
            this.container.addClass('active');
            this.overlay.addClass('active');
            $('body').addClass('poetry-screensaver-active');
            
            $(document).trigger('poetry-screensaver-started');
            
            setTimeout(() => {
                if (this.isScreensaverActive) {
                    this.startPoetrySequence();
                }
            }, 1000);
        }

        stopScreensaver() {
            if (!this.isScreensaverActive) {
                return;
            }
            
            this.log('Stopping screensaver mode');
            this.isScreensaverActive = false;
            this.isRunning = false;
            
            if (this.sequenceTimer) {
                clearTimeout(this.sequenceTimer);
                this.sequenceTimer = null;
            }
            
            this.container.removeClass('active');
            this.overlay.removeClass('active');
            $('body').removeClass('poetry-screensaver-active');
            
            this.container.empty();
            
            $(document).trigger('poetry-screensaver-stopped');
            
            if (this.settings.screensaverMode) {
                this.resetIdleTimer();
            }
        }

        startPoetrySequence() {
            if (this.isRunning || this.poems.length === 0 || !this.isScreensaverActive) {
                return;
            }
            
            this.log('Starting poetry sequence');
            this.isRunning = true;
            this.showNextLine();
        }

        showNextLine() {
            if (!this.isScreensaverActive || !this.isRunning) {
                return;
            }
            
            if (this.currentIndex >= this.poems.length) {
                this.log('Completed cycle, restarting...');
                this.currentIndex = 0;
                this.sequenceTimer = setTimeout(() => {
                    if (this.isScreensaverActive) {
                        this.showNextLine();
                    }
                }, this.settings.interval * 2000);
                return;
            }

            const poem = this.poems[this.currentIndex];
            this.createPoetryLine(poem);
            this.currentIndex++;

            this.sequenceTimer = setTimeout(() => {
                this.showNextLine();
            }, this.settings.interval * 1000);
        }

        createPoetryLine(text) {
            if (!this.isScreensaverActive) {
                return;
            }
            
            const lineElement = $('<div>')
                .addClass('poetry-line animated jf-sunday ')
                .addClass(`lang-${this.language}`)
                .text(text)
                .css({
                    'animation-duration': this.settings.speed + 's',
                    'animation-name': 'poetryFloat'
                });

            this.container.append(lineElement);

            // 動畫結束後自動移除
            setTimeout(() => {
                lineElement.remove();
            }, this.settings.speed * 1000);
        }

        adjustForMobile() {
            // 輕量版本 - 只確保文字居中
            if (this.isScreensaverActive) {
                this.container.find('.poetry-line').css({
                    'text-align': 'center',
                    'white-space': 'nowrap'
                });
            }
        }

        bindEvents() {
            // 活動偵測事件
            const activityEvents = [
                'mousedown', 'mousemove', 'keypress', 'scroll', 
                'touchstart', 'touchmove', 'click', 'wheel'
            ];
            
            activityEvents.forEach(event => {
                $(document).on(event + '.poetryScreensaver', () => {
                    if (this.settings.screensaverMode) {
                        this.resetIdleTimer();
                    }
                });
            });

            $(window).on('focus.poetryScreensaver blur.poetryScreensaver', () => {
                if (this.settings.screensaverMode) {
                    this.resetIdleTimer();
                }
            });

            $(window).on('resize.poetryScreensaver', () => {
                this.adjustForMobile();
            });

            document.addEventListener('visibilitychange', () => {
                if (document.hidden) {
                    if (this.isScreensaverActive) {
                        this.pauseAnimation();
                    }
                } else if (this.settings.screensaverMode) {
                    this.resetIdleTimer();
                }
            });

            // 鍵盤快捷鍵
            $(document).on('keydown.poetryScreensaver', (e) => {
                if (e.ctrlKey && e.shiftKey) {
                    switch(e.which) {
                        case 84: // T
                            e.preventDefault();
                            this.startScreensaver();
                            break;
                        case 80: // P
                            e.preventDefault();
                            if (this.isScreensaverActive) {
                                this.toggleAnimation();
                            }
                            break;
                        case 83: // S
                            e.preventDefault();
                            this.stopScreensaver();
                            break;
                    }
                } else if (this.isScreensaverActive) {
                    this.stopScreensaver();
                }
            });

            this.overlay.on('click.poetryScreensaver', () => {
                this.stopScreensaver();
            });

            this.container.on('click.poetryScreensaver', () => {
                this.stopScreensaver();
            });
        }

        pauseAnimation() {
            this.container.find('.poetry-line').addClass('paused');
        }

        resumeAnimation() {
            this.container.find('.poetry-line').removeClass('paused');
        }

        toggleAnimation() {
            const isPaused = this.container.find('.poetry-line.paused').length > 0;
            if (isPaused) {
                this.resumeAnimation();
            } else {
                this.pauseAnimation();
            }
        }

        updateSettings(newSettings) {
            this.settings = { ...this.settings, ...newSettings };
            this.setupContainers();
            if (this.isScreensaverActive) {
                this.stopScreensaver();
            }
        }

        destroy() {
            $(document).off('.poetryScreensaver');
            $(window).off('.poetryScreensaver');
            this.overlay.off('.poetryScreensaver');
            this.container.off('.poetryScreensaver');
            
            if (this.idleTimer) {
                clearTimeout(this.idleTimer);
                this.idleTimer = null;
            }
            if (this.sequenceTimer) {
                clearTimeout(this.sequenceTimer);
                this.sequenceTimer = null;
            }
            
            this.stopScreensaver();
        }

        getStatus() {
            const timeUntilScreensaver = this.idleTimer && !this.isScreensaverActive ? 
                Math.max(0, (this.lastActivity + this.settings.idleTime * 1000 - Date.now()) / 1000) : 0;
            
            return {
                isActive: this.isScreensaverActive,
                isRunning: this.isRunning,
                language: this.language,
                poemsCount: this.poems.length,
                timeUntilScreensaver: Math.round(timeUntilScreensaver),
                currentIndex: this.currentIndex,
                settings: this.settings
            };
        }
    }

    // DOM準備完成後初始化
    $(document).ready(function() {
        console.log('Poetry Screensaver: Lightweight version loading...');
        
        if (window.matchMedia('(prefers-reduced-motion: reduce)').matches) {
            console.log('Poetry Screensaver: Reduced motion detected, skipping');
            return;
        }

        if (typeof poetryScreensaverData === 'undefined') {
            console.error('Poetry Screensaver: Configuration not found');
            return;
        }

        setTimeout(() => {
            try {
                window.poetryScreensaver = new PoetryScreensaver();
                console.log('Poetry Screensaver: Lightweight version ready');
            } catch (error) {
                console.error('Poetry Screensaver: Initialization failed:', error);
            }
        }, 500); // 減少延遲時間
    });

    // 語言變化監聽
    $(document).on('wpml_lang_changed polylang_lang_changed', function(e, lang) {
        if (window.poetryScreensaver) {
            window.poetryScreensaver.loadPoetryData();
        }
    });

    // 全域控制API
    window.PoetryScreensaverControl = {
        start: function() {
            if (window.poetryScreensaver) {
                window.poetryScreensaver.startScreensaver();
                return true;
            }
            return false;
        },
        
        stop: function() {
            if (window.poetryScreensaver) {
                window.poetryScreensaver.stopScreensaver();
                return true;
            }
            return false;
        },
        
        status: function() {
            if (window.poetryScreensaver) {
                return window.poetryScreensaver.getStatus();
            }
            return null;
        },
        
        debug: function() {
            if (window.poetryScreensaver) {
                window.poetryScreensaverDebug = true;
                const status = window.poetryScreensaver.getStatus();
                console.log('=== Poetry Screensaver Status ===');
                console.log('Status:', status);
                console.log('Poems:', window.poetryScreensaver.poems.slice(0, 3));
                console.log('================================');
                return status;
            }
            return null;
        },
        
        testDefault: function() {
            if (window.poetryScreensaver) {
                window.poetryScreensaver.useDefaultPoems();
                window.poetryScreensaver.startScreensaver();
                return true;
            }
            return false;
        }
    };

})(jQuery);