<?php
/**
 * Plugin Name: Spot Slider
 * Plugin URI: https://your-website.com
 * Description: 將 Elementor sections 轉換成投影片式的單頁滾動效果
 * Version: 1.0.0
 * Author: Your Name
 * License: GPL v2 or later
 * Text Domain: spot-slider
 */

// 防止直接訪問
if (!defined('ABSPATH')) {
    exit;
}

class SpotSliderPlugin {
    
    public function __construct() {
        add_action('init', array($this, 'init'));
    }
    
    public function init() {
        // 註冊腳本和樣式
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        
        // 添加管理員選單
        add_action('admin_menu', array($this, 'admin_menu'));
        
        // 註冊設定
        add_action('admin_init', array($this, 'admin_init'));
        
        // 在前端頁面載入
        add_action('wp_footer', array($this, 'render_spot_slider'));
    }
    
    public function enqueue_scripts() {
        // 只在啟用的頁面載入，且不在後台或 Elementor 編輯模式
        $enabled_pages = get_option('spot_slider_pages', array());
        $current_page_id = get_the_ID();
        
        // 檢查是否在後台、預覽模式或 Elementor 編輯模式
        if (is_admin() || 
            is_preview() || 
            (isset($_GET['elementor-preview']) && $_GET['elementor-preview']) ||
            (class_exists('\Elementor\Plugin') && \Elementor\Plugin::$instance->editor->is_edit_mode()) ||
            (class_exists('\Elementor\Plugin') && \Elementor\Plugin::$instance->preview->is_preview_mode())) {
            return; // 不載入 Spot Slider
        }
        
        if (is_page() && in_array($current_page_id, $enabled_pages)) {

            add_action('wp_head', function() {
                echo '<style type="text/css" id="spot-slider-emergency-css">' . $this->get_css() . '</style>';
            }, 999);



            // 載入 CSS 和 JS（包含響應式檢測）
            wp_add_inline_script('wp-block-library', $this->get_js());
        }
    }
    
    public function admin_menu() {
        add_options_page(
            'Spot Slider 設定',
            'Spot Slider',
            'manage_options',
            'spot-slider',
            array($this, 'admin_page')
        );
    }
    
    public function admin_init() {
        register_setting('spot_slider_settings', 'spot_slider_pages');
        register_setting('spot_slider_settings', 'spot_slider_sections');
        register_setting('spot_slider_settings', 'spot_slider_animation_speed');
        register_setting('spot_slider_settings', 'spot_slider_show_nav');
    }
    
    public function admin_page() {
        if (isset($_POST['submit'])) {
            update_option('spot_slider_pages', $_POST['spot_slider_pages'] ?? array());
            update_option('spot_slider_sections', $_POST['spot_slider_sections'] ?? '');
            update_option('spot_slider_animation_speed', $_POST['spot_slider_animation_speed'] ?? '800');
            update_option('spot_slider_show_nav', isset($_POST['spot_slider_show_nav']) ? 1 : 0);
            echo '<div class="notice notice-success"><p>設定已儲存！</p></div>';
        }
        
        $pages = get_pages();
        $enabled_pages = get_option('spot_slider_pages', array());
        $sections = get_option('spot_slider_sections', 'sec-about,sec-whatson,sec-program,sec-team');
        $animation_speed = get_option('spot_slider_animation_speed', '800');
        $show_nav = get_option('spot_slider_show_nav', 1);
        ?>
        <div class="wrap">
            <h1>Spot Slider 設定</h1>
            <form method="post" action="">
                <?php wp_nonce_field('spot_slider_settings'); ?>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">啟用頁面</th>
                        <td>
                            <?php foreach ($pages as $page): ?>
                                <label style="display: block; margin-bottom: 5px;">
                                    <input type="checkbox" name="spot_slider_pages[]" value="<?php echo $page->ID; ?>" 
                                           <?php checked(in_array($page->ID, $enabled_pages)); ?>>
                                    <?php echo $page->post_title; ?>
                                </label>
                            <?php endforeach; ?>
                            <p class="description">選擇要啟用 Spot Slider 效果的頁面</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">Section ID 列表</th>
                        <td>
                            <input type="text" name="spot_slider_sections" value="<?php echo esc_attr($sections); ?>" 
                                   class="regular-text" placeholder="sec-about,sec-whatson,sec-program,sec-team">
                            <p class="description">輸入要轉換的 section ID，用逗號分隔</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">動畫速度 (毫秒)</th>
                        <td>
                            <input type="number" name="spot_slider_animation_speed" value="<?php echo esc_attr($animation_speed); ?>" 
                                   min="200" max="2000" step="100">
                            <p class="description">動畫持續時間，建議 500-1000 毫秒</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">顯示導航點</th>
                        <td>
                            <label>
                                <input type="checkbox" name="spot_slider_show_nav" value="1" <?php checked($show_nav, 1); ?>>
                                顯示右側導航點（建議開啟以便 debug）
                            </label>
                            <p class="description">勾選此項目會在右側顯示導航點，方便測試和切換區塊</p>
                        </td>
                    </tr>
                </table>
                
                <?php submit_button(); ?>
            </form>
            
            <div class="notice notice-info">
                <p><strong>使用說明：</strong></p>
                <ul>
                    <li>1. 選擇要啟用效果的頁面</li>
                    <li>2. 確保你的 Elementor sections 有正確的 ID (sec-about, sec-whatson, sec-program, sec-team)</li>
                    <li>3. 建議先勾選「顯示導航點」來測試效果</li>
                    <li>4. 儲存設定後到前端頁面查看效果</li>
                    <li>5. 滾輪只在左側 index 區域有效，右側 content 區域正常滾動</li>
                    <li>6. 每次只顯示一個區塊，其他區塊會隱藏</li>
                </ul>
            </div>
        </div>
        <?php
    }
    
    public function render_spot_slider() {
        // 檢查是否在後台、預覽模式或 Elementor 編輯模式
        if (is_admin() || 
            is_preview() || 
            (isset($_GET['elementor-preview']) && $_GET['elementor-preview']) ||
            (class_exists('\Elementor\Plugin') && \Elementor\Plugin::$instance->editor->is_edit_mode()) ||
            (class_exists('\Elementor\Plugin') && \Elementor\Plugin::$instance->preview->is_preview_mode())) {
            return; // 不載入 Spot Slider
        }
        
        $enabled_pages = get_option('spot_slider_pages', array());
        $current_page_id = get_the_ID();
        
        if (is_page() && in_array($current_page_id, $enabled_pages)) {
            $sections = get_option('spot_slider_sections', 'sec-about,sec-whatson,sec-program,sec-team');
            $sections_array = array_map('trim', explode(',', $sections));
            $show_nav = get_option('spot_slider_show_nav', 1);
            $animation_speed = get_option('spot_slider_animation_speed', '800');
            ?>
            <script>
            // 定義 SpotSlider 類別
            window.SpotSlider = class SpotSlider {
                constructor() {
                    this.slides = document.querySelectorAll('.spot-slide');
                    this.navDots = document.querySelectorAll('.spot-nav-dot');
                    this.currentIndex = 0;
                    this.isAnimating = false;

                    this.sectionMapping = {
                        'sec-about': 0,
                        'sec-whatson': 1, 
                        'sec-program': 2,
                        'sec-team': 3
                    };

                    
                    if (this.slides.length > 0) {
                        this.init();
                    }
                }
                
                init() {
                    // 綁定事件
                    this.bindEvents();
                    
                    // 初始化導航點
                    this.updateNavDots();
                    
                    // 顯示第一個 slide 的遮罩
                    if (this.slides.length > 0) {
                        this.showOverlays(this.slides[0]);
                    }

                    // *** 新增：錨點跳轉功能 ***
                    this.setupAnchorNavigation();
                    
                    // *** 新增：處理頁面載入時的錨點 ***
                    this.handleInitialHash();
                }

                // *** 新增方法：設置錨點導航 ***
                setupAnchorNavigation() {
                    // 監聽錨點變化
                    window.addEventListener('hashchange', () => {
                        this.handleHashChange();
                    });
                    
                    // 攔截錨點連結點擊
                    document.addEventListener('click', (e) => {
                        const link = e.target.closest('a[href^="#sec-"]');
                        if (link) {
                            e.preventDefault();
                            const hash = link.getAttribute('href').substring(1); // 移除 #
                            this.navigateToSection(hash);
                        }
                    });
                }
                
                // *** 新增方法：處理錨點變化 ***
                handleHashChange() {
                    const hash = window.location.hash.substring(1); // 移除 #
                    if (hash) {
                        this.navigateToSection(hash);
                    }
                }
                
                // *** 新增方法：處理頁面載入時的錨點 ***
                handleInitialHash() {
                    if (window.location.hash) {
                        // 延遲執行，確保 Spot Slider 完全初始化
                        setTimeout(() => {
                            this.handleHashChange();
                        }, 100);
                    }
                }
                
                // *** 新增方法：導航到指定 section ***
                navigateToSection(sectionId) {
                    const targetIndex = this.sectionMapping[sectionId];
                    if (targetIndex !== undefined && targetIndex !== this.currentIndex) {
                        console.log('錨點跳轉到:', sectionId, '索引:', targetIndex);
                        this.goToSlide(targetIndex);
                    }
                }
                            
                // 在 SpotSlider 的 bindEvents() 方法中修改滾輪事件處理

                bindEvents() {
                    // 定義只響應 spot slider 的區域
                    const indexSections = [
                        'about_index_sec',
                        'whatson_index_sec', 
                        'program_index_sec',
                        'team_index_sec'
                    ];
                    
                    // 定義內容區域（不應該觸發 spot slider）
                    const contentSections = [
                        'about_content_sec',
                        'whatson_content_sec',
                        'program_content_sec', 
                        'team_content_sec'
                    ];
                    
                    // 修復鍵盤事件處理 - 加入錯誤捕獲和更安全的處理
                    const handleKeydown = (e) => {
                        // 加入 try-catch 防止未捕獲的錯誤
                        try {
                            if (this.isAnimating) return;
                            
                            // 檢查是否有任何輸入框處於焦點狀態
                            const activeElement = document.activeElement;
                            if (activeElement && (
                                activeElement.tagName === 'INPUT' || 
                                activeElement.tagName === 'TEXTAREA' || 
                                activeElement.contentEditable === 'true'
                            )) {
                                return; // 如果有輸入框在焦點，不處理方向鍵
                            }
                            
                            switch(e.key) {
                                case 'ArrowUp':
                                    e.preventDefault();
                                    e.stopPropagation();
                                    this.goToPrevious();
                                    console.log('鍵盤上鍵觸發：上一個區塊');
                                    break;
                                case 'ArrowDown':
                                    e.preventDefault();
                                    e.stopPropagation();
                                    this.goToNext();
                                    console.log('鍵盤下鍵觸發：下一個區塊');
                                    break;
                            }
                        } catch (error) {
                            console.error('鍵盤事件處理錯誤:', error);
                            // 不要重新拋出錯誤，避免中斷其他功能
                        }
                    };
                    
                    // 使用更安全的事件綁定
                    document.addEventListener('keydown', handleKeydown, {
                        passive: false,
                        capture: true // 使用捕獲階段，更早攔截事件
                    });
                    
                    // 滑鼠滾輪事件處理 - 同樣加入錯誤捕獲
                    let wheelTimeout;
                    let isScrolling = false;
                    let accumulatedDelta = 0;
                    const deltaThreshold = 100;
                    const resetTimeout = 150;
                    
                    // 在 spot-slider.php 的 SpotSlider bindEvents() 方法中更新滾輪事件處理

                    const handleWheel = (e) => {
                        try {
                            // 檢查滑鼠是否在 content 區域上方
                            const target = e.target;
                            let isInContentArea = false;
                            let isInIndexArea = false;
                            
                            // 向上遍歷 DOM 樹，檢查是否在特定區域內
                            let currentElement = target;
                            while (currentElement && currentElement !== document.body) {
                                // 檢查是否在內容區域
                                if (contentSections.includes(currentElement.id)) {
                                    isInContentArea = true;
                                    console.log('滑鼠在 content 區域:', currentElement.id);
                                    break;
                                }
                                
                                // 檢查是否在 index 區域  
                                if (indexSections.includes(currentElement.id)) {
                                    isInIndexArea = true;
                                    console.log('滑鼠在 index 區域:', currentElement.id);
                                    break;
                                }
                                
                                // 特別檢查是否在可滾動的容器內 - 擴展所有 pane
                                if (currentElement.id === 'about-pane' ||
                                    currentElement.id === 'whatson-pane' || 
                                    currentElement.id === 'program-pane' ||
                                    currentElement.id === 'team-pane') {
                                    isInContentArea = true;
                                    console.log('滑鼠在可滾動容器:', currentElement.id);
                                    break;
                                }
                                
                                // 檢查是否在任何有 va-target 類別的容器內
                                if (currentElement.classList && currentElement.classList.contains('va-target')) {
                                    isInContentArea = true;
                                    console.log('滑鼠在 va-target 容器:', currentElement.id || currentElement.className);
                                    break;
                                }
                                
                                currentElement = currentElement.parentElement;
                            }
                            
                            // 如果在內容區域，檢查是否可以滾動
                            if (isInContentArea) {
                                // 尋找最近的可滾動祖先
                                let scrollableParent = target;
                                while (scrollableParent && scrollableParent !== document.body) {
                                    const computedStyle = window.getComputedStyle(scrollableParent);
                                    const overflowY = computedStyle.overflowY;
                                    
                                    if (overflowY === 'auto' || overflowY === 'scroll') {
                                        const scrollTop = scrollableParent.scrollTop;
                                        const scrollHeight = scrollableParent.scrollHeight;
                                        const clientHeight = scrollableParent.clientHeight;
                                        
                                        const canScrollDown = scrollTop < (scrollHeight - clientHeight - 5); // 5px 容錯
                                        const canScrollUp = scrollTop > 5; // 5px 容錯
                                        
                                        // 如果可以滾動，允許正常滾動行為
                                        if ((e.deltaY > 0 && canScrollDown) || (e.deltaY < 0 && canScrollUp)) {
                                            console.log('允許內容區域滾動 - 容器:', scrollableParent.id || scrollableParent.className);
                                            console.log('滾動狀態 - scrollTop:', scrollTop, 'scrollHeight:', scrollHeight, 'clientHeight:', clientHeight);
                                            return; // 讓瀏覽器處理正常滾動
                                        } else {
                                            console.log('內容區域已到達滾動邊界');
                                        }
                                        break;
                                    }
                                    scrollableParent = scrollableParent.parentElement;
                                }
                                
                                // 如果在內容區域但找不到可滾動的父元素，仍然允許事件傳播
                                console.log('在內容區域但無可滾動容器，允許正常處理');
                                return;
                            }
                            
                            // 如果不在 index 區域，不處理 spot slider
                            if (!isInIndexArea) {
                                console.log('滑鼠不在 index 區域，允許正常滾動');
                                return; // 讓瀏覽器處理正常滾動
                            }
                            
                            // 在 index 區域才處理 spot slider
                            e.preventDefault();
                            e.stopPropagation();
                            
                            if (this.isAnimating || isScrolling) return;
                            
                            // 累積滾輪數值
                            accumulatedDelta += Math.abs(e.deltaY);
                            
                            // 清除重置計時器
                            clearTimeout(wheelTimeout);
                            
                            // 檢查是否達到閾值
                            if (accumulatedDelta >= deltaThreshold) {
                                isScrolling = true;
                                
                                // 決定方向並切換
                                if (e.deltaY > 0) {
                                    this.goToNext();
                                    console.log('滾輪往下：切換到下一個區塊');
                                } else {
                                    this.goToPrevious();
                                    console.log('滾輪往上：切換到上一個區塊');
                                }
                                
                                // 重置累積值
                                accumulatedDelta = 0;
                                
                                // 防止連續觸發
                                setTimeout(() => {
                                    isScrolling = false;
                                }, 500);
                            } else {
                                // 如果沒達到閾值，設置重置計時器
                                wheelTimeout = setTimeout(() => {
                                    accumulatedDelta = 0;
                                    console.log('重置滾輪累積值');
                                }, resetTimeout);
                            }
                            
                            console.log('滾輪累積值:', accumulatedDelta, '/', deltaThreshold);
                            
                        } catch (error) {
                            console.error('滾輪事件處理錯誤:', error);
                            // 不要重新拋出錯誤
                        }
                    };
                    
                    document.addEventListener('wheel', handleWheel, { passive: false });
                    document.addEventListener('DOMMouseScroll', handleWheel, { passive: false });
                    
                    // 導航點點擊事件 - 也加入錯誤捕獲
                    this.navDots.forEach((dot, index) => {
                        dot.addEventListener('click', () => {
                            try {
                                if (this.isAnimating) return;
                                this.goToSlide(index);
                                console.log('導航點點擊：跳轉到區塊', index + 1);
                            } catch (error) {
                                console.error('導航點點擊錯誤:', error);
                            }
                        });
                    });
                }
                
                goToNext() {
                    if (this.currentIndex < this.slides.length - 1) {
                        this.goToSlide(this.currentIndex + 1, 'down');
                    }
                }
                
                goToPrevious() {
                    if (this.currentIndex > 0) {
                        this.goToSlide(this.currentIndex - 1, 'up');
                    }
                }
                
                goToSlide(index, direction = null) {
                    if (index === this.currentIndex || this.isAnimating) return;
                    
                    console.log('切換到 section', index + 1, '，方向:', direction);
                    
                    this.isAnimating = true;
                    const animationDuration = parseInt('<?php echo $animation_speed; ?>') || 800;
                    
                    const currentSlide = this.slides[this.currentIndex];
                    const nextSlide = this.slides[index];
                    
                    console.log('當前 slide:', currentSlide.id, '下一個 slide:', nextSlide.id);
                    
                    // 決定動畫方向
                    if (!direction) {
                        direction = index > this.currentIndex ? 'down' : 'up';
                    }
                    
                    // 隱藏當前 slide 的遮罩
                    this.hideOverlays(currentSlide);
                    
                    // 設置下一個 slide 的初始位置
                    if (direction === 'down') {
                        nextSlide.style.transform = 'translateY(100%)';
                    } else {
                        nextSlide.style.transform = 'translateY(-100%)';
                    }
                    nextSlide.style.opacity = '0';
                    nextSlide.style.zIndex = '2';
                    
                    // 強制重繪
                    nextSlide.offsetHeight;
                    
                    // 開始真正絲滑的動畫 - 一步到位
                    requestAnimationFrame(() => {
                        // 當前 slide 完全移出（一次到位）
                        if (direction === 'down') {
                            currentSlide.style.transform = 'translateY(-100%)';
                        } else {
                            currentSlide.style.transform = 'translateY(100%)';
                        }
                        currentSlide.style.opacity = '0';
                        currentSlide.classList.remove('active');
                        
                        // 下一個 slide 完全進入（一次到位）
                        nextSlide.style.transform = 'translateY(0)';
                        nextSlide.style.opacity = '1';
                        nextSlide.classList.add('active');
                        
                        // 顯示下一個 slide 的遮罩
                        this.showOverlays(nextSlide);
                        
                        console.log('真正絲滑動畫開始');
                    });
                    
                    // 動畫完成後
                    setTimeout(() => {
                        currentSlide.style.zIndex = '1';
                        this.currentIndex = index;
                        this.updateNavDots();
                        this.isAnimating = false;
                        console.log('動畫完成，當前顯示:', nextSlide.id);
                    }, animationDuration);
                }

                // *** 新增方法：更新瀏覽器錨點 ***
                updateBrowserHash() {
                    const sectionIds = Object.keys(this.sectionMapping);
                    const currentSectionId = sectionIds[this.currentIndex];
                    if (currentSectionId) {
                        // 使用 replaceState 避免觸發 hashchange 事件
                        const newUrl = window.location.pathname + window.location.search + '#' + currentSectionId;
                        history.replaceState(null, null, newUrl);
                    }
                }
                            
                // 修改 SpotSlider 中的遮罩控制函數，加入安全檢查
                showOverlays(slide) {
                    if (!slide) {
                        console.warn('無法顯示遮罩 - slide 為 null');
                        return;
                    }
                    
                    try {
                        const overlays = slide.querySelectorAll('.spot-overlay');
                        if (overlays && overlays.length > 0) {
                            overlays.forEach(overlay => {
                                if (overlay && overlay.style) {
                                    overlay.style.opacity = '1';
                                }
                            });
                            console.log('顯示遮罩，數量:', overlays.length);
                        } else {
                            console.log('在', slide.id, '中未找到遮罩元素');
                        }
                    } catch (error) {
                        console.error('顯示遮罩時發生錯誤:', error);
                    }
                }
                
                hideOverlays(slide) {
                    if (!slide) {
                        console.warn('無法隱藏遮罩 - slide 為 null');
                        return;
                    }
                    
                    try {
                        const overlays = slide.querySelectorAll('.spot-overlay');
                        if (overlays && overlays.length > 0) {
                            overlays.forEach(overlay => {
                                if (overlay && overlay.style) {
                                    overlay.style.opacity = '0';
                                }
                            });
                            console.log('隱藏遮罩，數量:', overlays.length);
                        } else {
                            console.log('在', slide.id, '中未找到遮罩元素');
                        }
                    } catch (error) {
                        console.error('隱藏遮罩時發生錯誤:', error);
                    }
                }

                updateNavDots() {
                    this.navDots.forEach((dot, index) => {
                        if (index === this.currentIndex) {
                            dot.classList.add('active');
                        } else {
                            dot.classList.remove('active');
                        }
                    });
                }
            };
            
            function createOverlaysForSection(section) {
                // 定義左右側區域的 ID 和對應的遮罩顏色
                const overlayConfig = {
                    // 左側 index 區域 - 純黑色遮罩
                    'about_index_sec': 'rgba(0, 0, 0, 0.4)',
                    'whatson_index_sec': 'rgba(0, 0, 0, 0.4)', 
                    'program_index_sec': 'rgba(0, 0, 0, 0.4)',
                    'team_index_sec': 'rgba(0, 0, 0, 0.4)',
                    
                    // 右側 content 區域 - 深藍灰色遮罩
                    'about_content_sec': 'rgba(0, 0, 0, 0.8)',
                    'whatson_content_sec': 'rgba(0, 0, 0, 0.8)',
                    'program_content_sec': 'rgba(0, 0, 0, 0.8)',
                    'team_content_sec': 'rgba(0, 0, 0, 0.8)'
                };
                
                // 為每個配置的區域創建遮罩
                Object.keys(overlayConfig).forEach(targetId => {
                    const targetElement = section.querySelector('#' + targetId);
                    if (targetElement) {
                        // 確保目標元素有相對定位
                        targetElement.style.position = 'relative';
                        
                        // 創建遮罩元素
                        const overlay = document.createElement('div');
                        overlay.className = 'spot-overlay';
                        overlay.style.cssText = `
                            position: absolute !important;
                            top: 0 !important;
                            left: 0 !important;
                            right: 0 !important;
                            bottom: 0 !important;
                            background: ${overlayConfig[targetId]} !important;
                            opacity: 0 !important;
                            transition: opacity <?php echo $animation_speed; ?>ms ease !important;
                            pointer-events: none !important;
                            z-index: 1 !important;
                        `;
                        
                        // 將遮罩添加到目標元素
                        targetElement.appendChild(overlay);
                        
                        // 確保內容在遮罩之上
                        const children = targetElement.children;
                        for (let i = 0; i < children.length; i++) {
                            if (children[i] !== overlay) {
                                children[i].style.position = 'relative';
                                children[i].style.zIndex = '2';
                            }
                        }
                        
                        console.log('已為', targetId, '創建遮罩，顏色:', overlayConfig[targetId]);
                    }
                });
            }

            function calculateAvailableWidth() {
                const sidebar = document.getElementById('va-sidebar');
                let sidebarWidth = 0;
                
                if (sidebar) {
                    const isDesktop = window.innerWidth >= 1400;
                    const isTablet = window.innerWidth >= 1199 && window.innerWidth <= 1399;
                    
                    if (isDesktop) {
                        // 桌機模式：側邊欄可見，需要扣除其寬度
                        const sidebarRect = sidebar.getBoundingClientRect();
                        sidebarWidth = sidebarRect.width;
                        console.log('桌機模式 - 側邊欄寬度:', sidebarWidth + 'px');
                    } else if (isTablet) {
                        // 平板模式：側邊欄隱藏，寬度為0
                        sidebarWidth = 0;
                        console.log('平板模式 - 側邊欄隱藏，使用全寬');
                    }
                }
                
                const availableWidth = window.innerWidth - sidebarWidth;
                console.log('可用寬度:', availableWidth + 'px');
                return availableWidth;
            }
            
            function initSpotSlider(sectionIds) {
                // 智慧型寬度計算邏輯

                function bindArrowClickEvents() {
                    const indexSections = [
                        'about_index_sec',
                        'whatson_index_sec', 
                        'program_index_sec'
                    ];
                    
                    indexSections.forEach(sectionId => {
                        const section = document.getElementById(sectionId);
                        if (section) {
                            section.addEventListener('click', function(e) {
                                // 檢查點擊位置是否在底部箭頭區域
                                const rect = section.getBoundingClientRect();
                                const clickY = e.clientY;
                                const sectionBottom = rect.bottom;
                                
                                // 如果點擊位置在底部 60px 範圍內（箭頭區域）
                                if (clickY >= sectionBottom - 60) {
                                    e.preventDefault();
                                    e.stopPropagation();
                                    
                                    // 觸發下一個 slide
                                    if (window.spotSliderInstance) {
                                        window.spotSliderInstance.goToNext();
                                        console.log('點擊箭頭區域，切換到下一個 slide');
                                    }
                                }
                            });
                        }
                    });
                }
            
                const availableWidth = calculateAvailableWidth();
                
                // 計算可用寬度
                //const availableWidth = window.innerWidth - sidebarWidth;
                console.log('可用寬度:', availableWidth + 'px');
                
                // 創建 slider 容器
                const container = document.createElement('div');
                container.className = 'spot-slider-container';
                
                let firstSection = null;
                const validSections = [];
                
                // 收集有效的 sections
                sectionIds.forEach((id, index) => {
                    const section = document.getElementById(id);
                    if (section) {
                        console.log('找到 section:', id);
                        if (!firstSection) firstSection = section;
                        validSections.push(section);
                    } else {
                        console.log('找不到 section:', id);
                    }
                });
                
                if (!firstSection || validSections.length === 0) {
                    console.log('Spot Slider: 沒有找到任何有效的 sections');
                    return;
                }
                
                console.log('找到', validSections.length, '個有效的 sections');
                
                // 設置 slider 容器樣式 - 固定定位在右邊
                const isDesktop = window.innerWidth >= 1400;

                container.style.position = 'fixed';
                container.style.top = '0';
                container.style.width = availableWidth + 'px';
                container.style.height = '100vh';
                container.style.overflow = 'hidden';
                container.style.zIndex = '999';

                if (isDesktop) {
                    // 桌機模式：容器在右側，左邊留給側邊欄
                    container.style.right = '0';
                    container.style.left = 'auto';
                } else {
                    // 平板模式：容器佔滿全寬
                    container.style.left = '0';
                    container.style.right = 'auto';
                }
                container.style.height = '100vh';
                container.style.overflow = 'hidden';
                container.style.zIndex = '999';
                
                // 將容器添加到 body
                document.body.appendChild(container);
                
                // 移動所有 sections 到容器中並設置樣式
                validSections.forEach((section, index) => {
                    // 設置 section 樣式
                    section.style.position = 'absolute';
                    section.style.top = '0';
                    section.style.left = '0';
                    section.style.width = '100%';
                    section.style.height = '100vh';
                    section.style.margin = '0';
                    section.style.boxSizing = 'border-box';
                    
                    // 添加 slider 類別
                    section.classList.add('spot-slide');
                    if (index === 0) {
                        section.classList.add('active');
                        section.style.opacity = '1';
                        section.style.transform = 'translateY(0)';
                        section.style.zIndex = '2';
                    } else {
                        section.style.opacity = '0';
                        section.style.transform = 'translateY(100%)';
                        section.style.zIndex = '1';
                    }
                    
                    // 創建遮罩覆蓋層
                    createOverlaysForSection(section);
                    
                    container.appendChild(section);
                    console.log('Section', index + 1, '(', section.id, ') 已移動到 slider 容器');
                });
                
                // 隱藏 #page 的內容，但不是整個 #page
                const pageElement = document.getElementById('page');
                if (pageElement) {
                    // 只隱藏 #content，不隱藏整個 #page
                    const contentElement = document.getElementById('content');
                    if (contentElement) {
                        contentElement.style.display = 'none';
                        console.log('已隱藏 #content');
                    }
                    
                    // 清空 spot-slider-container 如果它已經存在於 #page 內
                    const existingContainer = pageElement.querySelector('.spot-slider-container');
                    if (existingContainer && existingContainer !== container) {
                        existingContainer.remove();
                        console.log('移除了 #page 內的舊 slider 容器');
                    }
                }
                
                // 添加導航點（如果啟用）
                <?php if ($show_nav): ?>
                const nav = document.createElement('div');
                nav.className = 'spot-nav';
                validSections.forEach((section, index) => {
                    const dot = document.createElement('div');
                    dot.className = 'spot-nav-dot' + (index === 0 ? ' active' : '');
                    dot.setAttribute('data-target', index);
                    dot.title = sectionIds[index] || ('Section ' + (index + 1));
                    nav.appendChild(dot);
                });
                document.body.appendChild(nav);
                console.log('導航點已添加');
                <?php else: ?>
                console.log('導航點已隱藏');
                <?php endif; ?>
                
                // 監聽視窗大小改變，重新計算寬度
                function updateContainerSize() {
                    const newAvailableWidth = calculateAvailableWidth();
                    const currentIsDesktop = window.innerWidth >= 1400;
                    
                    container.style.width = newAvailableWidth + 'px';
                    
                    if (currentIsDesktop) {
                        // 桌機模式：容器在右側
                        container.style.right = '0';
                        container.style.left = 'auto';
                    } else {
                        // 平板模式：容器佔滿全寬
                        container.style.left = '0';
                        container.style.right = 'auto';
                    }
                    
                    console.log('容器已更新 - 寬度:', newAvailableWidth + 'px', '模式:', currentIsDesktop ? '桌機' : '平板');
                }

                window.addEventListener('resize', updateContainerSize);
                
                // 初始化 SpotSlider 類別
                try {
                    const slider = new SpotSlider();
                    window.spotSliderInstance = slider;
                    console.log('Spot Slider 初始化成功');
                    ['about_index_sec', 'whatson_index_sec', 'program_index_sec'].forEach(sectionId => {
                        const section = document.getElementById(sectionId);
                        if (section) {
                            section.addEventListener('click', function(e) {
                                const rect = section.getBoundingClientRect();
                                const clickY = e.clientY;
                                
                                // 如果點擊在底部 50px 範圍內（箭頭區域）
                                if (clickY >= rect.bottom - 50) {
                                    e.preventDefault();
                                    slider.goToNext();
                                    console.log('點擊箭頭，切換下一個 slide');
                                }
                            });
                        }
                    });
                    console.log('容器寬度:', availableWidth + 'px');
                    console.log('容器位置: fixed, right:0, top:0');
                    console.log('當前顯示區塊:', validSections[0].id);
                } catch (error) {
                    console.error('Spot Slider 初始化失敗:', error);
                }
            }
            
            document.addEventListener('DOMContentLoaded', function() {
                // 螢幕寬度檢測
                function checkScreenSize() {
                    return window.innerWidth >= 1199;
                }
                
                // 初始化 Spot Slider
                const sections = <?php echo json_encode($sections_array); ?>;
                let spotSliderInstance = null;
                let isInitialized = false;
                
                function initializeSpotSlider() {
                    if (!checkScreenSize()) {
                        console.log('螢幕寬度小於 575px，Spot Slider 已停用');
                        return;
                    }
                    
                    // 檢查是否有有效的 sections
                    let hasValidSections = false;
                    sections.forEach(id => {
                        if (document.getElementById(id)) {
                            hasValidSections = true;
                        }
                    });
                    
                    if (hasValidSections && !isInitialized) {
                        initSpotSlider(sections);
                        isInitialized = true;
                        console.log('Spot Slider 已初始化（桌面模式）');
                    } else if (!hasValidSections) {
                        console.log('Spot Slider: 找不到指定的 section IDs');
                    }
                }
                
                function destroySpotSlider() {
                    if (isInitialized) {
                        // 移除 spot slider 容器
                        const container = document.querySelector('.spot-slider-container');
                        if (container) {
                            // 將 sections 移回原位
                            const slides = container.querySelectorAll('.spot-slide');
                            const contentElement = document.getElementById('content');
                            
                            if (contentElement) {
                                slides.forEach(slide => {
                                    // 清除 spot slider 樣式
                                    slide.style.cssText = '';
                                    slide.classList.remove('spot-slide', 'active');
                                    
                                    // 移除遮罩
                                    const overlays = slide.querySelectorAll('.spot-overlay');
                                    overlays.forEach(overlay => overlay.remove());
                                    
                                    // 移回原始容器（這裡可能需要根據實際結構調整）
                                    contentElement.appendChild(slide);
                                });
                                
                                // 顯示原始內容
                                contentElement.style.display = '';
                            }
                            
                            // 移除容器和導航
                            container.remove();
                            const nav = document.querySelector('.spot-nav');
                            if (nav) nav.remove();
                        }
                        
                        isInitialized = false;
                        console.log('Spot Slider 已停用（行動模式）');
                    }
                }
                
                function handleResize() {
                    const isDesktop = checkScreenSize();
                    
                    if (isDesktop && !isInitialized) {
                        // 切換到桌面模式，初始化 slider
                        initializeSpotSlider();
                    } else if (!isDesktop && isInitialized) {
                        // 切換到行動模式，停用 slider
                        destroySpotSlider();
                    }
                }
                
                // 初始檢查
                initializeSpotSlider();
                
                // 監聽視窗大小改變
                window.addEventListener('resize', handleResize);
                
                console.log('響應式 Spot Slider 系統已啟動');

                
            });
            </script>
            <?php
        }
    }
    
    private function get_css() {
        $animation_speed = get_option('spot_slider_animation_speed', '800');
        $show_nav = get_option('spot_slider_show_nav', 1);
        $speed_ms = $animation_speed . 'ms';
        
        $nav_css = '';
        if ($show_nav) {
            $nav_css = "
            .spot-nav {
                position: fixed !important;
                right: 30px !important;
                top: 50vh !important;
                margin-top: -40px !important;
                z-index: 9999 !important;
                display: block !important;
                width: 20px !important;
                height: 80px !important;
                pointer-events: auto !important;
            }
            
            .spot-nav-dot {
                display: block !important;
                width: 12px !important;
                height: 12px !important;
                margin: 0 auto 15px auto !important;
                border-radius: 50% !important;
                background: rgba(255, 255, 255, 0.6) !important;
                cursor: pointer !important;
                transition: all 0.3s ease !important;
                position: relative !important;
                pointer-events: auto !important;
                box-sizing: border-box !important;
            }
            
            .spot-nav-dot:last-child {
                margin-bottom: 0 !important;
            }
            
            .spot-nav-dot:hover {
                background: rgba(255, 255, 255, 0.9) !important;
                transform: scale(1.1) !important;
            }
            
            .spot-nav-dot.active {
                background: #fff !important;
                transform: scale(1.2) !important;
                box-shadow: 0 0 10px rgba(255, 255, 255, 0.5) !important;
            }
            
            .spot-nav-dot::after {
                content: '' !important;
                position: absolute !important;
                top: 50% !important;
                left: 50% !important;
                transform: translate(-50%, -50%) !important;
                width: 20px !important;
                height: 20px !important;
                border: 1px solid rgba(255, 255, 255, 0.4) !important;
                border-radius: 50% !important;
                transition: all 0.3s ease !important;
                pointer-events: none !important;
                box-sizing: border-box !important;
            }
            
            .spot-nav-dot.active::after {
                border-color: #23e2d0 !important;
                border-width: 2px !important;
            }
            
            @media (max-width: 768px) {
                .spot-nav {
                    right: 20px !important;
                    width: 16px !important;
                    height: 64px !important;
                    margin-top: -32px !important;
                }
                
                .spot-nav-dot {
                    width: 10px !important;
                    height: 10px !important;
                    margin-bottom: 12px !important;
                }
                
                .spot-nav-dot::after {
                    width: 16px !important;
                    height: 16px !important;
                }
            }
            ";
        }
        
        return "
        .spot-slider-container {
            position: fixed !important;
            top: 0 !important;
            right: 0 !important;
            /* 寬度由 JavaScript 動態計算 */
            height: 100vh !important;
            overflow: hidden !important;
            z-index: 99 !important;
        }
        
        .spot-slide {
            position: absolute !important;
            top: 0 !important;
            left: 0 !important;
            width: 100% !important;
            height: 100vh !important;
            margin: 0 !important;
            box-sizing: border-box !important;
            opacity: 0 !important;
            transform: translateY(100%) !important;
            transition: all {$speed_ms} cubic-bezier(0.25, 0.46, 0.45, 0.94), 
                        opacity {$speed_ms} cubic-bezier(0.25, 0.46, 0.45, 0.94) !important;
            z-index: 1 !important;
            overflow-y: auto !important;
        }
        
        .spot-slide.active {
            opacity: 1 !important;
            transform: translateY(0) !important;
            z-index: 2 !important;
        }
        
        /* 確保目標元素有相對定位 */
        #about_index_sec,
        #whatson_index_sec,
        #program_index_sec,
        #team_index_sec,
        #about_content_sec,
        #whatson_content_sec,
        #program_content_sec,
        #team_content_sec {
            position: relative !important;
        }
        
        /* 左側 index 區域的半透明遮罩 */
        #about_index_sec::before,
        #whatson_index_sec::before,
        #program_index_sec::before,
        #team_index_sec::before {
            content: '' !important;
            position: absolute !important;
            top: 0 !important;
            left: 0 !important;
            right: 0 !important;
            bottom: 0 !important;
            background: rgba(0, 0, 0, 0.6) !important;
            transition: opacity {$speed_ms} ease !important;
            pointer-events: none !important;
            z-index: 1 !important;
        }
        
        /* 右側 content 區域的半透明遮罩 */
        #about_content_sec::before,
        #whatson_content_sec::before,
        #program_content_sec::before,
        #team_content_sec::before {
            content: '' !important;
            position: absolute !important;
            top: 0 !important;
            left: 0 !important;
            right: 0 !important;
            bottom: 0 !important;
            background: rgba(20, 20, 40, 0.7) !important;
            opacity: 0 !important;
            transition: opacity {$speed_ms} ease !important;
            pointer-events: none !important;
            z-index: 1 !important;
        }
        
        /* 當 section 變為 active 時，遮罩淡入 */
        .spot-slide.active #about_index_sec::before,
        .spot-slide.active #whatson_index_sec::before,
        .spot-slide.active #program_index_sec::before,
        .spot-slide.active #team_index_sec::before,
        .spot-slide.active #about_content_sec::before,
        .spot-slide.active #whatson_content_sec::before,
        .spot-slide.active #program_content_sec::before,
        .spot-slide.active #team_content_sec::before {
            opacity: 1 !important;
        }
        
        /* 確保內容文字在遮罩之上且為白色 */
        #about_index_sec > *,
        #whatson_index_sec > *,
        #program_index_sec > *,
        #team_index_sec > *,
        #about_content_sec > *,
        #whatson_content_sec > *,
        #program_content_sec > *,
        #team_content_sec > * {
            position: relative !important;
            z-index: 2 !important;
        }

        @media (min-width: 1199px) {

            /* 只隱藏 #content，不隱藏整個 #page */
            #content {
                display: none !important;
            }
            
            /* 確保 #page 本身保持可見 */
            #page {
                display: block !important;
            }

        }
        
        
        
        {$nav_css}
        ";
    }
    
    private function get_js() {
        // 不需要額外的 JS，因為都已經在 render_spot_slider() 中定義了
        return '';
    }
}

// 初始化外掛
new SpotSliderPlugin();
?>