// va-common.js
// 專案共用函式庫，包含內容載入和響應式判斷等通用邏輯

(function() {
  function qs(sel, ctx) { return (ctx || document).querySelector(sel); }
  function qsa(sel, ctx) { return Array.prototype.slice.call((ctx || document).querySelectorAll(sel)); }

  // 檢查是否為手機模式（小於等於 1199px）
  // 這是唯一的判斷來源
  function isMobile() {
    return window.matchMedia('(max-width: 1199px)').matches;
  }

  // 取得目標元素的 key（slug 或 ID）
  function findKey(el) {
    var slug = el.getAttribute('data-va-slug');
    if (slug) return { type: 'slug', value: slug };
    var id = el.getAttribute('data-va-id');
    if (id) return { type: 'id', value: id };
    var m = parseById(el.id);
    if (m) {
      var n = parseInt(m.key, 10);
      if (!isNaN(n)) return { type: 'id', value: String(n) };
      return { type: 'slug', value: m.key };
    }
    return null;
  }

  // 根據 ID 解析 key 和 targetId
  function parseById(id) {
    if (!id || id.indexOf('va--') !== 0) return null;
    var p = id.split('--');
    if (p.length < 3) return null;
    return { key: p[1], targetId: p[2] };
  }

  // 根據 data-va-target 或 ID 尋找目標區塊
  function findTarget(el) {
    var sel = el.getAttribute('data-va-target');
    if (sel) { try { return qs(sel); } catch (e) { return null; } }
    var m = parseById(el.id);
    if (m && m.targetId) { return qs('#' + m.targetId); }
    return null;
  }

  // 顯示讀取中的狀態
  function setBusy(el, isBusy) {
    if (el) {
      el.classList.toggle('va-loading', isBusy);
      el.setAttribute('aria-busy', isBusy);
    }
  }

  // 處理 AJAX 內容載入與渲染
  function fetchAndInject(key, targetEl) {
    if (!key || !targetEl) { return Promise.reject('Missing key or target element.'); }

    const data = {
      'action': 'va_fetch',
      'nonce': window.va_ajax.nonce,
    };
    if (key.type === 'slug') { data.slug = key.value; }
    if (key.type === 'id') { data.id = key.value; }

    setBusy(targetEl, true);

    return fetch(window.va_ajax.url, {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: new URLSearchParams(data)
      })
      .then(response => response.json())
      .then(result => {
        if (result.success) {
          const contentHTML = `
            ${result.data.media ? `<figure class="wp-block-image">${result.data.media}</figure>` : ''}
            <h2>${result.data.title}</h2>
            ${result.data.content}
          `;
          targetEl.innerHTML = contentHTML;
        } else {
          targetEl.innerHTML = `<p class="va-error">${result.data.message}</p>`;
        }
        setBusy(targetEl, false);
        return result;
      })
      .catch(error => {
        console.error('AJAX 內容載入失敗:', error);
        setBusy(targetEl, false);
        targetEl.innerHTML = `<p class="va-error">載入失敗，請稍後再試。</p>`;
        return Promise.reject(error);
      });
  }

  // 將通用函式暴露給全域物件
  window.vaCommon = {
    qs,
    qsa,
    isMobile,
    findKey,
    findTarget,
    fetchAndInject,
  };

})();

