/**
 * VA Core - Vanishing Acts 展覽前端核心系統
 * 整合所有互動功能：內容載入、響應式導航、Modal 系統等
 * Version: 1.1.0
 */

(function() {
  'use strict';
  
  // =============================================================================
  // 核心工具函數
  // =============================================================================
  
  function qs(selector, context) { 
    return (context || document).querySelector(selector); 
  }
  
  function qsa(selector, context) { 
    return Array.prototype.slice.call((context || document).querySelectorAll(selector)); 
  }
  
  // 響應式斷點檢測
  const breakpoints = {
    mobile: () => window.matchMedia('(max-width: 1199px)').matches,
    tablet: () => window.matchMedia('(min-width: 1199px) and (max-width: 1399px)').matches,
    desktop: () => window.matchMedia('(min-width: 1400px)').matches
  };
  
  // =============================================================================
  // 內容管理系統
  // =============================================================================
  
  class ContentManager {
    constructor() {
      this.cache = new Map();
    }
    
    /**
     * 解析元素的識別資訊
     */
    parseElementId(element) {
      // 優先檢查 data 屬性
      const slug = element.getAttribute('data-va-slug');
      if (slug) return { type: 'slug', value: slug };
      
      const id = element.getAttribute('data-va-id');
      if (id) return { type: 'id', value: id };
      
      // 解析 ID 格式: va--{slugOrId}--{targetId}
      const elementId = element.id;
      if (elementId && elementId.indexOf('va--') === 0) {
        const parts = elementId.split('--');
        if (parts.length >= 2) {
          const key = parts[1];
          const numericKey = parseInt(key, 10);
          return isNaN(numericKey) 
            ? { type: 'slug', value: key }
            : { type: 'id', value: String(numericKey) };
        }
      }
      
      return null;
    }
    
    /**
     * 尋找目標容器
     */
    findTargetContainer(element) {
      // 檢查 data-va-target 屬性
      const selector = element.getAttribute('data-va-target');
      if (selector) {
        try { 
          return qs(selector); 
        } catch(e) { 
          console.warn('Invalid target selector:', selector);
          return null; 
        }
      }
      
      // 從 ID 解析目標
      const elementId = element.id;
      if (elementId && elementId.indexOf('va--') === 0) {
        const parts = elementId.split('--');
        if (parts.length >= 3) {
          return qs('#' + parts[2]);
        }
      }
      
      return null;
    }
    
    /**
     * 設置載入狀態
     */
    setLoadingState(container, isLoading) {
      if (!container) return;
      
      if (isLoading) {
        container.setAttribute('aria-busy', 'true');
        container.innerHTML = `<div class="va-loading">${window.vaData?.i18n?.loading || 'Loading…'}</div>`;
      } else {
        container.removeAttribute('aria-busy');
      }
    }
    
    /**
     * 渲染內容到容器
     */
    renderContent(container, data) {
      if (!container || !data) return;
      
      const html = (data.media || '') + (data.content || '');
      container.innerHTML = html;
      
      // 觸發自定義事件
      container.dispatchEvent(new CustomEvent('va:rendered', { detail: data }));
      document.dispatchEvent(new CustomEvent('va:carousel:init', { 
        detail: { root: container }
      }));
      
      console.log('Content rendered successfully');
    }
    
    /**
     * 通過 AJAX 獲取並注入內容
     */
    async fetchAndInject(key, container) {
      if (!key || !container) return;
      
      // 檢查緩存
      const cacheKey = `${key.type}:${key.value}`;
      if (this.cache.has(cacheKey)) {
        this.renderContent(container, this.cache.get(cacheKey));
        return;
      }
      
      this.setLoadingState(container, true);
      
      try {
        const formData = new FormData();
        formData.append('action', 'va_fetch');
        formData.append('nonce', window.vaData?.nonce || '');
        formData.append(key.type, key.value);
        
        const response = await fetch(window.vaData?.ajax_url || '/wp-admin/admin-ajax.php', {
          method: 'POST',
          credentials: 'same-origin',
          body: formData
        });
        
        const result = await response.json();
        
        if (!result || !result.success) {
          throw new Error(result?.data?.message || 'Content not found');
        }
        
        // 緩存結果
        this.cache.set(cacheKey, result.data);
        this.renderContent(container, result.data);
        
      } catch (error) {
        console.error('Failed to fetch content:', error);
        container.innerHTML = `<div class="va-empty">${window.vaData?.i18n?.not_found || 'Content not found'}</div>`;
      } finally {
        this.setLoadingState(container, false);
      }
    }
  }
  
  // =============================================================================
  // Modal 系統
  // =============================================================================
  
  class ModalManager {
    constructor() {
      this.modal = null;
      this.isInitialized = false;
    }
    
    /**
     * 確保 Modal 已創建並初始化
     */
    ensureModal() {
      if (!this.modal) {
        this.modal = qs('#va-modal');
        if (!this.modal) {
          this.createModal();
        }
      }
      
      if (!this.isInitialized) {
        this.bindEvents();
        this.isInitialized = true;
      }
      
      return this.modal;
    }
    
    /**
     * 創建 Modal 結構
     */
    createModal() {
      this.modal = document.createElement('div');
      this.modal.id = 'va-modal';
      this.modal.className = 'va-modal';
      this.modal.hidden = true;
      this.modal.setAttribute('aria-hidden', 'true');
      this.modal.setAttribute('role', 'dialog');
      this.modal.setAttribute('aria-modal', 'true');
      
      this.modal.innerHTML = `
        <div class="va-modal__backdrop" data-va-modal-close></div>
        <div class="va-modal__panel">
          <button class="va-modal__close" aria-label="Close" data-va-modal-close>&times;</button>
          <div id="va-modal-body" class="va-modal__body va-target entry-content"></div>
          <div class="va-modal-nav" style="display: none;">
            <button class="va-modal-prev" aria-label="Previous">‹ Prev</button>
            <button class="va-modal-next" aria-label="Next">Next ›</button>
          </div>
          <div class="va-modal-dots"></div>
        </div>
      `;
      
      document.body.appendChild(this.modal);
    }
    
    /**
     * 綁定 Modal 事件
     */
    bindEvents() {
      if (!this.modal) return;
      
      // 關閉按鈕事件
      const closeBtn = this.modal.querySelector('.va-modal__close');
      const backdrop = this.modal.querySelector('.va-modal__backdrop');
      
      if (closeBtn) {
        closeBtn.addEventListener('click', (e) => {
          e.preventDefault();
          e.stopPropagation();
          this.closeModal();
        });
      }
      
      if (backdrop) {
        backdrop.addEventListener('click', (e) => {
          e.preventDefault();
          e.stopPropagation();
          this.closeModal();
        });
      }
      
      // ESC 鍵關閉
      document.addEventListener('keydown', (e) => {
        if (e.key === 'Escape' && !this.modal.hidden) {
          this.closeModal();
        }
      });
      
      // 防止滾動穿透
      const panel = this.modal.querySelector('.va-modal__panel');
      if (panel) {
        this.preventScrollThrough(panel);
      }
    }
    
    /**
     * 防止觸控滾動穿透
     */
    preventScrollThrough(panel) {
      panel.addEventListener('touchmove', (e) => {
        const isScrollable = panel.scrollHeight > panel.clientHeight;
        const isAtTop = panel.scrollTop === 0;
        const isAtBottom = panel.scrollTop + panel.clientHeight >= panel.scrollHeight;
        
        const touchY = e.touches[0].clientY;
        const lastTouchY = panel._lastTouchY || touchY;
        const deltaY = touchY - lastTouchY;
        
        if ((!isScrollable) || 
            (deltaY > 0 && isAtTop) || 
            (deltaY < 0 && isAtBottom)) {
          e.preventDefault();
        }
        
        panel._lastTouchY = touchY;
      }, { passive: false });
      
      panel.addEventListener('touchstart', (e) => {
        panel._lastTouchY = e.touches[0].clientY;
      });
    }
    
    /**
     * 打開 Modal
     */
    openModal(context = '') {
      const modal = this.ensureModal();
      
      // 防止背景滾動
      document.body.classList.add('va-modal-open');
      document.documentElement.style.overflow = 'hidden';
      
      // 設置上下文
      this.setModalContext(context);
      
      // 動畫效果
      modal.hidden = false;
      modal.style.opacity = '0';
      modal.style.visibility = 'hidden';
      
      // 強制重新計算樣式
      modal.offsetHeight;
      
      requestAnimationFrame(() => {
        requestAnimationFrame(() => {
          modal.style.opacity = '';
          modal.style.visibility = '';
          modal.setAttribute('aria-hidden', 'false');
        });
      });
      
      return qs('#va-modal-body');
    }
    
    /**
     * 關閉 Modal
     */
    closeModal() {
      if (!this.modal) return;
      
      this.modal.style.transition = 'opacity 0.4s ease, visibility 0.4s ease';
      this.modal.style.opacity = '0';
      this.modal.style.visibility = 'hidden';
      
      setTimeout(() => {
        this.modal.hidden = true;
        this.modal.setAttribute('aria-hidden', 'true');
        
        // 清理樣式
        this.modal.style.opacity = '';
        this.modal.style.visibility = '';
        this.modal.style.transition = '';
        
        // 恢復背景滾動
        document.body.classList.remove('va-modal-open');
        document.documentElement.style.overflow = '';
        
        this.setModalContext('');
      }, 400);
    }
    
    /**
     * 設置 Modal 上下文
     */
    setModalContext(context) {
      if (!this.modal) return;
      
      this.modal.dataset.ctx = context || '';
      
      // What's On 以外的上下文清空導航點
      if (context !== 'whatson') {
        const dots = this.modal.querySelector('.va-modal-dots');
        if (dots) dots.innerHTML = '';
      }
    }
  }
  
  // =============================================================================
  // 響應式導航系統
  // =============================================================================
  
  class NavigationManager {
    constructor() {
      this.init();
    }
    
    init() {
      this.initMobileBottomNav();
      this.initTabletHamburger();
      this.initMobileSectionSwitcher();
    }
    
    /**
     * 手機版底部導航
     */
    initMobileBottomNav() {
      const nav = qs('#va-mob-nav');
      if (!nav) return;
      
      // More 按鈕事件
      nav.addEventListener('click', (e) => {
        const moreBtn = e.target.closest('[data-open-more]');
        if (moreBtn) {
          e.preventDefault();
          this.openMore();
        }
      });
      
      // 關閉 More 面板事件
      qsa('[data-close-more]').forEach(el => {
        el.addEventListener('click', (e) => {
          e.preventDefault();
          this.closeMore();
        });
      });
      
      // More 面板背景點擊關閉
      const morePanel = qs('#va-more');
      if (morePanel) {
        morePanel.addEventListener('click', (e) => {
          if (e.target.matches('.va-more__backdrop')) {
            this.closeMore();
          }
        });
      }
      
      // ESC 鍵關閉
      document.addEventListener('keydown', (e) => {
        if (e.key === 'Escape') {
          this.closeMore();
        }
      });
      
      // 滾動導航
      qsa('[data-scroll]').forEach(link => {
        link.addEventListener('click', (e) => {
          const href = link.getAttribute('href') || '';
          if (href.startsWith('#') && href.length > 1) {
            e.preventDefault();
            if (link.hasAttribute('data-close-more')) {
              this.closeMore();
            }
            this.smoothScrollTo(href);
          }
        });
      });
    }
    
    /**
     * 平板版漢堡選單
     */
    initTabletHamburger() {
      const hamburger = qs('#va-hamburger');
      const sidebar = qs('#va-sidebar');
      const overlay = qs('#va-sidebar-overlay');
      
      if (!hamburger || !sidebar || !overlay) return;
      
      const toggleSidebar = () => {
        const isOpen = sidebar.classList.contains('is-open');
        if (isOpen) {
          this.closeSidebar();
        } else {
          this.openSidebar();
        }
      };
      
      // 事件綁定
      hamburger.addEventListener('click', toggleSidebar);
      overlay.addEventListener('click', () => this.closeSidebar());
      
      // 側邊欄連結點擊後關閉
      sidebar.addEventListener('click', (e) => {
        if (e.target.tagName === 'A' && window.innerWidth <= 1399) {
          this.closeSidebar();
        }
      });
      
      // ESC 鍵和視窗大小改變處理
      document.addEventListener('keydown', (e) => {
        if (e.key === 'Escape' && sidebar.classList.contains('is-open')) {
          this.closeSidebar();
        }
      });
      
      window.addEventListener('resize', () => {
        if (window.innerWidth >= 1400) {
          this.closeSidebar();
        }
      });
    }
    
    /**
     * 手機版區段切換器
     */
    initMobileSectionSwitcher() {
      if (!breakpoints.mobile()) {
        console.log('非手機模式，跳過區段切換器');
        return;
      }
      
      console.log('初始化手機版區段切換器');
      
      // 區段映射關係
      const sectionMapping = {
        '#sec-about': 'about_index_sec',
        '#sec-whatson': 'whatson_index_sec',
        '#sec-program': 'program_index_sec',
        '#sec-team': 'team_index_sec'
      };
      
      const allSections = Object.values(sectionMapping);
      const allSectionElements = allSections.map(id => qs('#' + id)).filter(Boolean);
      
      if (allSectionElements.length === 0) {
        console.log('找不到任何可切換的區段元素');
        return;
      }
      
      // 隱藏所有區段
      const hideAllSections = () => {
        allSectionElements.forEach(section => {
          section.style.display = 'none';
        });
      };
      
);
        link = qsa('.va-link[id]').find(el => regex.test(el.id));
      }
      
      if (link) {
        return this.contentManager.parseElementId(link);
      }
      
      return null;
    }
  }
  
  // =============================================================================
  // 高度同步系統
  // =============================================================================
  
  class LayoutManager {
    constructor() {
      this.init();
    }
    
    init() {
      this.syncHeights();
      window.addEventListener('resize', () => this.syncHeights());
    }
    
    /**
     * 同步容器高度
     */
    syncHeights() {
      if (breakpoints.mobile()) return;
      
      const selectors = [
        '#about_content_sec',
        '#about_index_sec',
        '#whatson_index_sec',
        '#whatson_content_sec',
        '#program_content_sec',
        '#program_index_sec',
        '#team_content_sec',
        '#team_index_sec'
      ];
      
      selectors.forEach(selector => {
        const element = qs(selector);
        if (element) {
          element.style.height = '100vh';
          element.style.maxHeight = '100vh';
        }
      });
    }
  }
  
  // =============================================================================
  // 主控制器
  // =============================================================================
  
  class VACore {
    constructor() {
      this.contentManager = new ContentManager();
      this.modalManager = new ModalManager();
      this.navigationManager = new NavigationManager();
      this.whatsOnManager = new WhatsOnManager();
      this.programmeManager = new ProgrammeManager();
      this.autoLoader = new AutoLoader(this.contentManager);
      this.layoutManager = new LayoutManager();
      
      this.init();
    }
    
    init() {
      this.bindContentLinks();
      this.bindProgrammeRows();
      this.autoLoader.loadDefaults();
      this.injectMobileStyles();
      
      console.log('VA Core initialized successfully');
    }
    
    /**
     * 綁定內容連結
     */
    bindContentLinks() {
      qsa('.va-link').forEach(link => {
        link.addEventListener('click', (e) => {
          e.preventDefault();
          
          const target = this.contentManager.findTargetContainer(link);
          const key = this.contentManager.parseElementId(link);
          
          // 手機版特殊處理
          if (breakpoints.mobile() && target && this.shouldOpenModal(target)) {
            const context = target.id === 'whatson-pane' ? 'whatson' : '';
            const modalBody = this.modalManager.openModal(context);
            if (modalBody && key) {
              this.contentManager.fetchAndInject(key, modalBody);
            }
            return;
          }
          
          // 一般處理
          if (target && key) {
            this.contentManager.fetchAndInject(key, target);
          }
        });
      });
    }
    
    /**
     * 綁定程式列表行點擊
     */
    bindProgrammeRows() {
      qsa('.va-prog-index .va-prog-row').forEach(row => {
        row.addEventListener('click', () => {
          qsa('.va-prog-index .va-prog-row').forEach(r => {
            r.classList.remove('is-active');
          });
          row.classList.add('is-active');
        });
      });
    }
    
    /**
     * 檢查是否應開啟 Modal
     */
    shouldOpenModal(target) {
      const modalTargets = ['about-pane', 'whatson-pane', 'program-pane'];
      return modalTargets.includes(target.id);
    }
    
    /**
     * 注入手機版導航樣式
     */
    injectMobileStyles() {
      if (qs('#va-mobile-nav-styles')) return;
      
      const styles = `
        .va-mob-nav__item.va-nav-active {
          background: #23e2d0 !important;
          color: #000 !important;
          transform: translateY(-2px);
          box-shadow: 0 4px 8px rgba(236, 253, 0, 0.3);
        }
        
        .va-mob-nav__item.va-nav-active span {
          color: #000;
        }
        
        .va-mob-nav__item {
          transition: all 0.3s ease;
        }
        
        @media (max-width: 1199px) {
          #about_content_sec,
          #whatson_content_sec, 
          #program_content_sec,
          #team_content_sec {
            display: none !important;
          }
          
          #about_index_sec,
          #whatson_index_sec,
          #program_index_sec, 
          #team_index_sec {
            width: 100% !important;
            max-width: 100% !important;
            min-height: 100vh;
          }
        }
      `;
      
      const styleElement = document.createElement('style');
      styleElement.id = 'va-mobile-nav-styles';
      styleElement.textContent = styles;
      document.head.appendChild(styleElement);
    }
  }
  
  // =============================================================================
  // 初始化和公開 API
  // =============================================================================
  
  // 等待 DOM 就緒或 Elementor 就緒
  function initializeVA() {
    const vaCore = new VACore();
    
    // 公開 API 給外部使用
    window.VA = {
      core: vaCore,
      contentManager: vaCore.contentManager,
      modalManager: vaCore.modalManager,
      navigationManager: vaCore.navigationManager,
      
      // 向後兼容的方法
      fetchAndInject: (key, target) => vaCore.contentManager.fetchAndInject(key, target),
      openModal: (context) => vaCore.modalManager.openModal(context),
      closeModal: () => vaCore.modalManager.closeModal(),
      ensureModal: () => vaCore.modalManager.ensureModal(),
      isMobile: breakpoints.mobile
    };
  }
  
  // 初始化邏輯
  if (window.elementorFrontend && elementorFrontend.hooks) {
    elementorFrontend.hooks.addAction('frontend/element_ready/global', initializeVA);
  } else if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initializeVA);
  } else {
    initializeVA();
  }
  
})();
