(function(){
  function qs(sel,ctx){ return (ctx||document).querySelector(sel); }
  function qsa(sel,ctx){ return Array.prototype.slice.call((ctx||document).querySelectorAll(sel)); }

  // 支援 id="va--{slugOrId}--{targetId}"
  function parseById(id){
    if(!id || id.indexOf('va--')!==0) return null;
    var p = id.split('--');
    if(p.length < 3) return null;
    return { key: p[1], targetId: p[2] };
  }

  function findTarget(el){
    var sel = el.getAttribute('data-va-target');
    if(sel){ try{ return qs(sel); }catch(e){ return null; } }
    var m = parseById(el.id);
    if(m && m.targetId){ return qs('#'+m.targetId); }
    return null;
  }

  function findKey(el){
    var slug = el.getAttribute('data-va-slug');
    if(slug) return { type:'slug', value: slug };
    var id = el.getAttribute('data-va-id');
    if(id) return { type:'id', value: id };
    var m = parseById(el.id);
    if(m){
      var n = parseInt(m.key,10);
      if(!isNaN(n)) return { type:'id', value: String(n) };
      return { type:'slug', value: m.key };
    }
    return null;
  }

  function setBusy(target, busy){
    if(!target) return;
    if(busy){
      target.setAttribute('aria-busy','true');
      target.innerHTML = '<div class="va-loading">'+(window.vaData&&vaData.i18n&&vaData.i18n.loading||'Loading…')+'</div>';
    } else {
      target.removeAttribute('aria-busy');
    }
  }

  function render(target, data){
    if(!target) return;
    // 你目前的版面：不顯示標題，只回 media + content
    var html = (data.media||'') + (data.content||'');
    target.innerHTML = html;
    target.dispatchEvent(new CustomEvent('va:rendered', { detail:data }));

    document.dispatchEvent(new CustomEvent('va:carousel:init', { detail: { root: target }}));
  }

  function fetchAndInject(key, target){
    if(!key || !target) return;
    setBusy(target,true);
    var fd = new FormData();
    fd.append('action','va_fetch');
    fd.append('nonce', (window.vaData && vaData.nonce) || '');
    if(key.type==='slug') fd.append('slug', key.value); else fd.append('id', key.value);

    fetch(vaData.ajax_url, { method:'POST', credentials:'same-origin', body:fd })
      .then(function(r){ return r.json(); })
      .then(function(res){
        if(!res || !res.success) throw new Error((res && res.data && res.data.message) || 'Not found');
        render(target, res.data);
      })
      .catch(function(){
        target.innerHTML = '<div class="va-empty">'+(vaData.i18n&&vaData.i18n.not_found||'Content not found')+'</div>';
      })
      .finally(function(){ setBusy(target,false); });
  }

  // --- Mobile modal helpers ---
  function isMobile(){ return window.matchMedia('(max-width: 1199px)').matches; }

    
  // 同時確保 ensureModal 中的事件綁定正確
  function ensureModal(){
      var m = document.querySelector('#va-modal');
      if(!m){
          m = document.createElement('div');
          m.id='va-modal';
          m.className='va-modal';
          m.hidden = true;
          m.setAttribute('aria-hidden','true');
          m.setAttribute('role','dialog');
          m.setAttribute('aria-modal','true');
          m.innerHTML = ''
              + '<div class="va-modal__backdrop" data-va-modal-close></div>'
              + '<div class="va-modal__panel">'
              + '  <button class="va-modal__close" aria-label="Close" data-va-modal-close>&times;</button>'
              + '  <div id="va-modal-body" class="va-modal__body va-target entry-content"></div>'
              + '</div>';
          document.body.appendChild(m);
      }
      
      if (!m._vaBound) {
          // 修改事件綁定：使用 addEventListener 代替事件委託
          const closeBtn = m.querySelector('.va-modal__close');
          const backdrop = m.querySelector('.va-modal__backdrop');
          
          if (closeBtn) {
              closeBtn.addEventListener('click', function(e){
                  e.preventDefault();
                  e.stopPropagation();
                  closeModal();
              });
          }
          
          if (backdrop) {
              backdrop.addEventListener('click', function(e){
                  e.preventDefault();
                  e.stopPropagation();
                  closeModal();
              });
          }
          
          // ESC 鍵關閉
          document.addEventListener('keydown', function(e){ 
              if(e.key==='Escape' && !m.hidden) {
                  closeModal(); 
              }
          });
          
          // 防止滾動穿透
          const panel = m.querySelector('.va-modal__panel');
          if (panel) {
              panel.addEventListener('touchmove', function(e) {
                  const isScrollable = panel.scrollHeight > panel.clientHeight;
                  const isAtTop = panel.scrollTop === 0;
                  const isAtBottom = panel.scrollTop + panel.clientHeight >= panel.scrollHeight;
                  
                  const touchY = e.touches[0].clientY;
                  const lastTouchY = panel._lastTouchY || touchY;
                  const deltaY = touchY - lastTouchY;
                  
                  if ((!isScrollable) || 
                      (deltaY > 0 && isAtTop) || 
                      (deltaY < 0 && isAtBottom)) {
                      e.preventDefault();
                  }
                  
                  panel._lastTouchY = touchY;
              }, { passive: false });
              
              panel.addEventListener('touchstart', function(e) {
                  panel._lastTouchY = e.touches[0].clientY;
              });
          }
          
          m._vaBound = true;
      }
      
      return m;
  }
  // 在 va-map.js 中，替換現有的 openModal 和 closeModal 函數：

  function openModal(ctx){
    var m = ensureModal();
    
    // 防止背景滾動
    document.body.classList.add('va-modal-open');
    document.documentElement.style.overflow = 'hidden';
    
    // 設置 context
    setModalContext(ctx || '');
    
    // 先移除 hidden 但保持透明
    m.hidden = false;
    m.style.opacity = '0';
    m.style.visibility = 'hidden';
    
    // 強制重新計算樣式
    m.offsetHeight;
    
    // 使用 requestAnimationFrame 確保動畫在下一幀觸發
    requestAnimationFrame(() => {
        requestAnimationFrame(() => {
            // 清除 inline styles，讓 CSS 的 :not([hidden]) 生效
            m.style.opacity = '';
            m.style.visibility = '';
            m.setAttribute('aria-hidden','false');
        });
    });
    
    return document.querySelector('#va-modal-body');
  }
  

  function closeModal(){
      var m = document.querySelector('#va-modal'); 
      if(!m) return;
      
      // 簡化關閉動畫：只控制最外層的透明度和可見性
      m.style.transition = 'opacity 0.4s ease, visibility 0.4s ease';
      m.style.opacity = '0';
      m.style.visibility = 'hidden';
      
      // 等待動畫完成後隱藏並清理
      setTimeout(() => {
          m.hidden = true; 
          m.setAttribute('aria-hidden','true');
          
          // 清理 inline styles
          m.style.opacity = '';
          m.style.visibility = '';
          m.style.transition = '';
          
          // 恢復背景滾動
          document.body.classList.remove('va-modal-open');
          document.documentElement.style.overflow = '';
          
          setModalContext('');
      }, 400);
  }


  // 綁點擊
  function bind(){
    qsa('.va-link').forEach(function(el){
      el.addEventListener('click', function(e){
        e.preventDefault();
        var target = findTarget(el);
        var key = findKey(el);

        // 手機：如果目標是 #about-pane 或 #va-pane-A（What's On 桌機容器），改走彈窗
        if (
          isMobile() && target && 
          (target.id === 'about-pane' || target.id === 'whatson-pane' || target.id === 'program-pane')
        ){
          // 只有 What's On 要顯示導覽
          target = openModal(target.id === 'whatson-pane' ? 'whatson' : '');
        }

        if(target && key){ fetchAndInject(key, target); }
      });
    });

    qsa('.va-prog-index .va-prog-row').forEach(function(row){
      row.addEventListener('click', function(){
        qsa('.va-prog-index .va-prog-row').forEach(function(r){ r.classList.remove('is-active'); });
        row.classList.add('is-active');
      });
    });

  }

  function setModalContext(ctx){
    var m = document.querySelector('#va-modal');
    if(!m) return;
    m.dataset.ctx = ctx || '';
    // 若不是 whatson，清空點點
    if (ctx !== 'whatson') {
      var dots = m.querySelector('.va-modal-dots');
      if (dots) dots.innerHTML = '';
    }
  }

  // === Programme Tabs ===
  function initProgrammeTabs(){
    // 僅桌機顯示 tabs：手機維持現在的清單 + 彈窗體驗
    if (window.matchMedia('(max-width: 1199px)').matches) return;

    var host = document.querySelector('#program-index[data-prog-tabs]');
    if (!host) return;

    var list = host.querySelector('.va-prog-index');
    var rows = Array.prototype.slice.call(host.querySelectorAll('.va-prog-index .va-prog-row'));
    if (!rows.length) return;

    // 收集資料夾標籤（來自每列的 data-va-tags，逗號分隔）
    var tagSet = new Set();
    rows.forEach(function(r){
      (r.getAttribute('data-va-tags')||'')
        .split(',').map(function(s){return s.trim();})
        .filter(Boolean).forEach(function(t){ tagSet.add(t); });
    });

    // 若沒有任何標籤就不產生 tabs
    if (!tagSet.size) return;

    // 產生 tabs 列（All + 各標籤）
    var tabsBar = host.querySelector('.va-prog-tabs');
    var rawTags = Array.from(tagSet);

    // 將資料結構改成 { key, label }，key 用來比對，label 用來顯示
    var tabs = [{ key: 'all', label: 'All' }].concat(
      rawTags.map(function(t){
        return { key: t.toLowerCase(), label: t };
      })
    );

    tabsBar.innerHTML = tabs.map(function(t,i){
      var active = i===0 ? ' is-active' : '';
      var ar = i===0 ? 'true' : 'false';
      return '<button class="va-tab text-file'+active+'" role="tab" aria-selected="'+ar+'" data-filter="'+t.key+'">' +
             '  <i class="fa fa-paperclip" aria-hidden="true"></i> ' +
             t.label.replace(/-/g,' ').replace(/_/g,' ') +
             '</button>';
    }).join('');

    // 點擊套用篩選
    tabsBar.addEventListener('click', function(e){
      var btn = e.target.closest('.va-tab'); if(!btn) return;
      tabsBar.querySelectorAll('.va-tab').forEach(function(b){
        b.classList.remove('is-active'); b.setAttribute('aria-selected','false');
      });
      btn.classList.add('is-active'); btn.setAttribute('aria-selected','true');
      apply(btn.getAttribute('data-filter'));
    });

    // 套用篩選（比對時一律轉小寫，避免大小寫問題）
    function apply(filter){
      var fl = (filter || 'all').toLowerCase();
      rows.forEach(function(r){
        var tags = (r.getAttribute('data-va-tags')||'')
          .split(',')
          .map(function(s){ return s.trim().toLowerCase(); })
          .filter(Boolean);
        var show = (fl === 'all') || tags.indexOf(fl) > -1;
        (r.parentElement || r).style.display = show ? '' : 'none';
      });
    }


    apply('all');
  }


  // 自動載入（手機略過 about / whatson 的桌機容器）
  function autoLoadDefaults(){
    qsa('.va-target').forEach(function(target){
      if (isMobile() && (target.id === 'about-pane' || target.id === 'whatson-pane' || target.id === 'program-pane')) return;

      var defSlug = target.getAttribute('data-va-default-slug');
      var defId   = target.getAttribute('data-va-default-id');
      if (defSlug || defId){
        var key = defSlug ? {type:'slug', value:defSlug} : {type:'id', value:defId};
        fetchAndInject(key, target);
        return;
      }
      if (target.id){
        var link = qs('.va-link[data-va-target="#'+target.id+'"][data-va-default="true"]')
               || qs('.va-link[data-va-target="#'+target.id+'"]');
        if (!link){
          var re = new RegExp('^va--(.+?)--'+target.id+'$');
          link = qsa('.va-link[id]').find(function(el){ return re.test(el.id); });
        }
        if (link){
          var key2 = findKey(link);
          if (key2) fetchAndInject(key2, target);
        }
      }
    });
  }

  function init(){ bind(); autoLoadDefaults(); initProgrammeTabs(); }
  if (window.elementorFrontend && elementorFrontend.hooks) {
    elementorFrontend.hooks.addAction('frontend/element_ready/global', init);
  } else if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else { init(); }

  // 簡化的高度設定函數 - 桌機時設為 70vh
  function $1(s){ return document.querySelector(s); }
  function syncAboutPaneHeight(){
    // 只在桌機（非手機）模式下執行
    if (isMobile()) return;
    
    var elements = [
      $1('#about_content_sec'),
      $1('#about_index_sec'),
      $1('#whatson_index_sec'), 
      $1('#whatson_index_sec'),
      $1('#program_content_sec'), 
      $1('#program_index_sec'),
      $1('#team_content_sec'), 
      $1('#team_index_sec'),
    ];
    
    elements.forEach(function(el) {
      if (el) {
        el.style.height = '100vh';
        el.style.maxHeight = '100vh';
      }
    });
  }

  function initSync(){
    syncAboutPaneHeight();
    // 監聽視窗大小變化，在桌機/手機模式切換時重新設定
    window.addEventListener('resize', syncAboutPaneHeight);
  }
  
  if (window.elementorFrontend && elementorFrontend.hooks) {
    elementorFrontend.hooks.addAction('frontend/element_ready/global', initSync);
  } else if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initSync);
  } else { initSync(); }

  // ⬇️ 匯出最小 API，給 va-mobile.js 使用
  window.VA = Object.assign(window.VA||{}, {
    fetchAndInject: fetchAndInject,
    openModal: openModal,
    closeModal: closeModal,
    ensureModal: ensureModal,
    isMobile: isMobile
  });
})();