<?php
/**
 * Plugin Name: Vanishing Acts Custom
 * Description: 專案「Vanishing Acts」的公用客製化外掛。集中放置前端互動、短碼、整合程式等。
 * Version: 1.0.1
 * Author: 8f-2 Studio
 * Text Domain: vanishing-acts
 */

// 在檔案最前面加入全域錯誤處理
add_action('wp_footer', function() {
    if (!is_admin() && !is_elementor_edit_mode()) {
        ?>
        <script>
        // 全域錯誤捕獲
        window.addEventListener('error', function(event) {
            if (event.error && event.error.message) {
                const message = event.error.message;
                if (message.includes('ControlLooksLikePasswordCredentialField') ||
                    message.includes('vanishing-layer') ||
                    message.includes('vanishing-overlay')) {
                    console.warn('捕獲並忽略已知錯誤:', message);
                    event.preventDefault();
                    return true;
                }
            }
        });
        
        // 未處理的 Promise 錯誤
        window.addEventListener('unhandledrejection', function(event) {
            if (event.reason && event.reason.message) {
                const message = event.reason.message;
                if (message.includes('ControlLooksLikePasswordCredentialField')) {
                    console.warn('捕獲並忽略 Promise 錯誤');
                    event.preventDefault();
                }
            }
        });
        </script>
        <?php
    }
}, 1);

if ( ! defined('ABSPATH') ) exit;

define('VA_CUSTOM_VERSION', '1.1.31');
define('VA_CUSTOM_SLUG', 'vanishing-acts-custom');
/**
 * Polylang 語言切換 CSS 控制
 * 根據當前語言在 body 添加對應的 class，控制 .lang-zh、.lang-en 和 .lang-es 的顯示隱藏
 */
class TFAMPolylangCSS {
    
    public function __construct() {
        add_action('wp_head', array($this, 'add_language_css'));
        add_filter('body_class', array($this, 'add_language_body_class'));
    }
    
    /**
     * 在 <head> 中添加 CSS 樣式
     */
    public function add_language_css() {
        ?>
        <style type="text/css">
        /* Polylang 語言切換 CSS - 高優先級規則 */
        .lang-zh, .lang-en, .lang-es {
            display: none !important;
        }
        
        /* 中文環境：只顯示中文內容，明確隱藏其他語言 */
        body.polylang-zh .lang-zh {
            display: block !important;
        }
        body.polylang-zh .lang-en,
        body.polylang-zh .lang-es {
            display: none !important;
        }
        
        /* 英文環境：只顯示英文內容，明確隱藏其他語言 */
        body.polylang-en .lang-en {
            display: block !important;
        }
        body.polylang-en .lang-zh,
        body.polylang-en .lang-es {
            display: none !important;
        }
        
        /* 西班牙語環境：只顯示西班牙語內容，明確隱藏其他語言 */
        body.polylang-es .lang-es {
            display: block !important;
        }
        body.polylang-es .lang-zh,
        body.polylang-es .lang-en {
            display: none !important;
        }
        
        /* 如果 Polylang 未啟用，預設顯示中文，隱藏其他語言 */
        body:not([class*="polylang-"]) .lang-zh {
            display: block !important;
        }
        body:not([class*="polylang-"]) .lang-en,
        body:not([class*="polylang-"]) .lang-es {
            display: none !important;
        }
        </style>
        <?php
    }
    
    /**
     * 在 body class 中添加語言識別 class
     */
    public function add_language_body_class($classes) {
        // 檢查 Polylang 是否啟用
        if (function_exists('pll_current_language')) {
            $current_lang = pll_current_language();
            if ($current_lang) {
                $classes[] = 'polylang-' . $current_lang;
            }
        }
        
        return $classes;
    }
}

// 初始化
new TFAMPolylangCSS();


// 允許在 Elementor 中使用更多 HTML 標籤和屬性
add_filter('wp_kses_allowed_html', 'allow_custom_html_attributes', 10, 2);

function allow_custom_html_attributes($tags, $context) {
    if ($context === 'post') {
        // 允許 a 標籤的自定義屬性
        $tags['a']['data-va-slug'] = true;
        $tags['a']['data-va-target'] = true;
        $tags['a']['class'] = true;
        
        // 允許 img 標籤的更多屬性
        $tags['img']['decoding'] = true;
        $tags['img']['loading'] = true;
    }
    return $tags;
}

/* =========================================================================
 * 檢查是否在 Elementor 編輯模式的輔助函數
 * ========================================================================= */
function is_elementor_edit_mode() {
    // 檢查是否在 Elementor 編輯模式
    if (isset($_GET['elementor-preview'])) {
        return true;
    }
    
    // 檢查 Elementor 的其他編輯相關參數
    if (isset($_GET['elementor_library']) || isset($_GET['ver']) || isset($_GET['action'])) {
        if (isset($_SERVER['REQUEST_URI']) && strpos($_SERVER['REQUEST_URI'], 'elementor') !== false) {
            return true;
        }
    }
    
    // 檢查是否載入了 Elementor 編輯器
    if (class_exists('\Elementor\Plugin')) {
        $elementor = \Elementor\Plugin::instance();
        if (method_exists($elementor, 'editor') && is_object($elementor->editor)) {
            if (method_exists($elementor->editor, 'is_edit_mode') && $elementor->editor->is_edit_mode()) {
                return true;
            }
        }
        if (method_exists($elementor, 'preview') && is_object($elementor->preview)) {
            if (method_exists($elementor->preview, 'is_preview_mode') && $elementor->preview->is_preview_mode()) {
                return true;
            }
        }
    }
    
    return false;
}

/* =========================================================================
 * VA Map：索引連結 → 讀取 vanishing_act 文章 → 注入指定容器
 * ========================================================================= */

if ( ! function_exists('va_map_register_cpt') ) {
  function va_map_register_cpt(){
    $labels = [
      'name'          => __('Vanishing Acts','vanishing-acts'),
      'singular_name' => __('Vanishing Act','vanishing-acts'),
      'menu_name'     => __('Vanishing Acts','vanishing-acts'),
      'add_new_item'  => __('Add New Vanishing Act','vanishing-acts'),
      'edit_item'     => __('Edit Vanishing Act','vanishing-acts'),
    ];
    register_post_type('vanishing_act', [
      'labels'       => $labels,
      'public'       => true,
      'show_in_rest' => true,
      'has_archive'  => false,
      'supports'     => ['title','editor','excerpt','thumbnail','revisions'],
      'rewrite'      => ['slug'=>'va'],
    ]);
  }
  add_action('init','va_map_register_cpt');
}

if ( ! function_exists('va_map_enqueue') ) {
  function va_map_enqueue(){
    // 在管理後台或 Elementor 編輯模式下不載入
    if ( is_admin() || is_elementor_edit_mode() ) return;
    
    $url = plugin_dir_url(__FILE__);
    wp_enqueue_style ('va-map-style',  $url.'assets/va-map.css', [], VA_CUSTOM_VERSION);
    wp_enqueue_script('va-map-script', $url.'assets/va-map.js',  [], VA_CUSTOM_VERSION, true);
    wp_enqueue_script('va-mobile-script', $url.'assets/va-mobile.js',  [], VA_CUSTOM_VERSION, true);
    wp_localize_script('va-map-script','vaData',[
      'ajax_url'=>admin_url('admin-ajax.php'),
      'nonce'   =>wp_create_nonce('va_ajax'),
      'i18n'    =>[
        'loading'=>__('Loading…','vanishing-acts'),
        'not_found'=>__('Content not found','vanishing-acts'),
      ]
    ]);
  }
  add_action('wp_enqueue_scripts','va_map_enqueue');
}

if ( ! function_exists('va_map_fetch') ) {
  function va_map_fetch(){
    check_ajax_referer('va_ajax','nonce');

    $slug = isset($_POST['slug']) ? sanitize_title(wp_unslash($_POST['slug'])) : '';
    $id   = isset($_POST['id']) ? absint($_POST['id']) : 0;

    if (!$slug && !$id){
      wp_send_json_error(['message'=>__('Missing identifier','vanishing-acts')], 400);
    }

    if ($id){
      $post = get_post($id);
    } else {
      $post = get_page_by_path($slug, OBJECT, 'vanishing_act');
    }

    if (!$post || 'vanishing_act' !== $post->post_type || 'publish' !== $post->post_status){
      wp_send_json_error(['message'=>__('Not found','vanishing-acts')], 404);
    }

    $title   = get_the_title($post);
    //$content = apply_filters('the_content', $post->post_content);
    $content = do_shortcode( apply_filters('the_content', $post->post_content) );
    $thumb   = get_the_post_thumbnail($post, 'large', ['alt'=>esc_attr($title)]);

    wp_send_json_success([
      'title'  => $title,
      'content'=> $content,
      'media'  => $thumb,
      'link'   => get_permalink($post),
    ]);
  }
  add_action('wp_ajax_va_fetch','va_map_fetch');
  add_action('wp_ajax_nopriv_va_fetch','va_map_fetch');
}

// 載入 Google 字型（只在非編輯模式）
add_action('wp_enqueue_scripts', function() {
  if ( ! is_admin() && ! is_elementor_edit_mode() ) {
    wp_enqueue_style(
      'va-google-fonts',
      'https://fonts.googleapis.com/css2?family=Fira+Mono&family=Playfair+Display&family=Space+Mono&family=Bebas+Neue&family=Inconsolata&display=swap',
      [],
      null
    );
  }
});
